import json
import logging
import os
from functools import wraps

import requests

import gevent

MINUTE_SECONDS = 60
HOUR_SECONDS = 60 * 60
DAY_SECONDS = 24 * HOUR_SECONDS

JUGGLER_CLIENT_URL = "https://localhost:31579/events"


class InvalidServiceConfiguration(Exception):
    def __init__(self, name):
        super(InvalidServiceConfiguration, self).__init__(
            f"Invalid service configuration, missing '{name}' option")


def get_environ(name):
    try:
        key_name = f"WALLE_{name}"
        return os.environ[key_name]
    except KeyError as e:
        raise InvalidServiceConfiguration(name) from e


def with_timeout(timeout):
    def wrapper(func):
        @wraps(func)
        def wrapped(*args, **kwargs):
            with gevent.Timeout(timeout):
                return func(*args, **kwargs)
        return wrapped
    return wrapper


def send_checks(check_result):
    response = requests.post(JUGGLER_CLIENT_URL, json=check_result.to_dict())
    result = response.json()

    if response.status_code != requests.status_codes.codes.OK:
        logging.error("Failed to send checks to juggler: %s", result["message"])
    elif not result["success"]:
        for event, event_result in zip(check_result.events, result["events"]):
            if event_result["code"] != 200:
                logging.error("Failed to send event %s to juggler: %s", event.service, event_result["error"])
    else:
        logging.info("Successfully sent %s events to juggler", len(check_result.events))


def print_checks(check_result):
    logging.info(json.dumps(check_result.to_dict(), indent=4, sort_keys=True))


class JugglerEvent(object):
    def __init__(self, service, status, description):
        self.service = service
        self.status = status
        self.description = description

    def to_dict(self, **kwargs):
        return dict(
            service=self.service,
            status=self.status,
            description=self.description,
            **kwargs
        )


class CheckResult(object):
    def __init__(self, source, events, tags):
        """
        :type source: str
        :type events: List[JugglerEvent]
        :type tags: List[str]
        """

        self.source = source
        self.events = events
        self.tags = tags

    def to_dict(self):
        return dict(
            source=self.source,
            events=[
                event.to_dict(tags=self.tags) for event in self.events
            ],
        )


class Monitor(object):
    def __init__(self, name, setup, checks, sender=send_checks):
        self.name = name

        self._setup = setup
        self._checks = checks
        self._sender = sender

    def run(self):
        check_result = self.execute_all_checks()
        self._sender(check_result)

    def execute_all_checks(self):
        ctx = self._setup()
        results = [self.execute_check(name, check, ctx) for name, check in self._checks.items()]
        return CheckResult(source=self.name, events=results, tags=["mongodb", "wall-e.srv", "wall-e-mongodb"])

    @staticmethod
    def execute_check(name, check, ctx):
        status, description = check(ctx)
        return JugglerEvent(service=name, status=status, description=description)
