import argparse
from database.s3_mds.client import S3Client, upload, fetch, remove, list_names, list_buckets, new_bucket, \
    get_access_key, get_service_id, _HelpTokenUrl


def _parse_args():
    parser = argparse.ArgumentParser(description="Wall-E uploader for S3-MDS.")
    parser.add_argument("-c", "--config", help="configuration file path", required=False)

    parser.add_argument("-T", "--testing", action="store_true", help="Use testing S3-MDS service", required=False)
    parser.add_argument("-u", "--url", help="url for s3-mds service", required=False)
    parser.add_argument("-k", "--access-key", help="access key for s3-mds service", required=False)
    parser.add_argument("-s", "--secret-key", help="secret key for s3-mds service", required=False)

    action = parser.add_subparsers(dest="action", required=True)

    upload_action = action.add_parser("upload", help="Upload file to a specified bucket")
    upload_action.set_defaults(handle=upload)
    upload_action.add_argument("bucket", help="bucket to upload file to")
    upload_action.add_argument("name", help="key/name to use for data on s3")
    upload_action.add_argument("filename", nargs='?', help="file to upload")

    fetch_action = action.add_parser("fetch", help="Fetch specified file from a bucket")
    fetch_action.set_defaults(handle=fetch)
    fetch_action.add_argument("bucket", help="bucket to fetch file from")
    fetch_action.add_argument("name", help="key/name to fetch from s3")
    fetch_action.add_argument("filename", nargs='?', help="save data to filename")

    remove_action = action.add_parser("remove", help="Remove specified file from a bucket")
    remove_action.set_defaults(handle=remove)
    remove_action.add_argument("bucket", help="bucket to remove file from")
    remove_action.add_argument("name", help="key/name to remove from s3")

    list_buckets_action = action.add_parser("list-names", help="List files in a bucket")
    list_buckets_action.set_defaults(handle=list_names)
    list_buckets_action.add_argument("bucket", help="bucket name")

    list_buckets_action = action.add_parser("list-buckets", help="List buckets")
    list_buckets_action.set_defaults(handle=list_buckets)

    new_bucket_action = action.add_parser("new-bucket", help="Create new bucket")
    new_bucket_action.set_defaults(handle=new_bucket)
    new_bucket_action.add_argument("bucket", help="bucket name")

    get_key_action = action.add_parser("get-key", help="Get access key to access your service in S3-MDS")
    get_key_action.set_defaults(handle=get_access_key)
    get_key_action.add_argument("token", help="oauth token for S3-MDS service")
    get_key_action.add_argument("service_id", help="Service ID in ABC")
    get_key_action.add_argument("role", choices=("reader", "admin", "owner"), help="User role in S3 service")

    get_service_id_action = action.add_parser(
        "get-service-id",
        help="Fetch your service's id from ABC by slug",
        description=(
            "Fetch abc service id from ABC for you. Get your oauth token for staff/abc at "
            "https://oauth.yandex-team.ru/authorize?response_type=token&client_id=9e9702c0b7f54152ac339989d9039ccd. "
            "Slug if your abc service is usually the last component of the url, e.g. for "
            "https://abc.yandex-team.ru/services/Wall-E/ the slug is 'Wall-E'"
        )
    )
    get_service_id_action.set_defaults(handle=get_service_id)
    get_service_id_action.add_argument("token", help="oauth token for abc/staff service (try get-token-url --help)")
    get_service_id_action.add_argument("slug", help="slug if your abc service")

    get_token_url_action = action.add_parser("get-token-url",
                                             help="Print url to obtain the oauth token for the system",
                                             )
    get_token_url_action.add_argument("system", choices=("s3-mds", "abc"), action=_HelpTokenUrl,
                                      help="Print url to obtain the oauth token for the system")

    return parser.parse_args()


def main():
    args = _parse_args()

    # use prod s3 service if not specified.
    if args.testing:
        args.url = S3Client.TEST_URL
    elif not args.url:
        args.url = S3Client.PROD_URL

    args.client = S3Client(args.access_key, args.secret_key, args.url or S3Client.TEST_URL)
    args.handle(args)


if __name__ == '__main__':
    main()
