from debian.changelog import Changelog
import datetime
import re
import logging


def bump_changelog(changelog_path, messages, version=None, author=None, distributions=None):
    with open(changelog_path) as changelog_file:
        changelog = Changelog(changelog_file)

    if author:
        if not re.match(r'^([\w\d_.-]+\s)+<[\w\d_.-]+@[\w\d.-]+>$', author):
            logging.debug("'Author' name string must include name, surname and e-mail in a form like")
            logging.debug("Name Surname <name.surname@company.org>")
            logging.debug("Look for examples in the changelog file {}".format(changelog_path))
            raise ValueError("author")
    else:
        author = changelog.author

    if version is None:
        last_version = sorted(changelog.versions)[-1]
        version = _bump_minor_version(last_version.full_version)

    changelog.new_block(
        package=changelog.package, version=version,
        distributions=distributions or changelog.distributions,
        urgency=changelog.urgency, author=author,
        date=datetime.datetime.utcnow().strftime("%a, %d %b %Y %H:%M:%S +0000"),
        changes=[""] + ["  * {}".format(msg) for msg in messages] + [""]
    )

    changelog_data = str(changelog)

    logging.info("Adding changelog record for new package version %s to %s", version, changelog_path)
    with open(changelog_path, "w") as changelog_file:
        changelog_file.write(changelog_data)

    return version


def _bump_minor_version(full_version):
    version_parts = full_version.split(".")

    minor_version = version_parts[-1].split("-")
    minor_version[0] = str(int(minor_version[0]) + 1)

    return ".".join(version_parts[:-1] + ["-".join(minor_version)])
