import logging
import os

import click
from . import changelog, arcadia
from .credentials import get_sandbox_token, get_release_author, CRD_RELEASE_AUTHOR_ENV, CRD_SANDBOX_OAUTH_TOKEN_ENV
from .requirements import set_minimum_dependency_version
from .sandbox import build_deb_bundle_task, build_checks_bundle_task, release_to_pypi_task, build_to_ya_tool_task,\
    DebRepo
from .tools import release_message, setup_logging
from .versions import bump_version_file


@click.group()
@click.option("-v", "--verbose", count=True, help="Don't be so silent, print something")
def release(verbose=0):
    setup_logging(verbose)


def version_bump_options(with_no_bump):
    bump_major = click.option("--major", 'bump', flag_value='major', help="Bump major release version")
    bump_minor = click.option("--minor", 'bump', flag_value='minor', help="Bump minor release version")
    bump_build = click.option("--build", 'bump', flag_value='build', default=True, help="Bump build number")
    bump_prestable = click.option("--prestable", 'bump', flag_value='prerelease', help="Bump prestable release version")
    no_bump = click.option("--no-bump", 'bump', flag_value='', help="Do not bump release version, build and release as-is")

    def add_bump_options(cmd):
        if with_no_bump:
            cmd = no_bump(cmd)
        return bump_major(bump_minor(bump_build(bump_prestable(cmd))))

    return add_bump_options


@release.command("release-sdk", help="Release Wall-E.sdk and Wall-E.cli to the outer world")
@version_bump_options(with_no_bump=True)
@click.option("--sdk-version", required=False, help="Version dotted string for new SDK release")
@click.option("-a", "--author", envvar=CRD_RELEASE_AUTHOR_ENV, required=False,
              help="Author name in form 'Name Surname <name.surname@company.org>'")
@click.option("-m", "--messages", required=False, multiple=True)
@click.option("--client-version", required=False, help="Version dotted string for new Client release")
@click.option("--client-messages", required=False, multiple=True)
@click.option("--sandbox-oauth-token", envvar=CRD_SANDBOX_OAUTH_TOKEN_ENV, required=False)
@click.option("-w/-W", "--wait/--no-wait", default=True, help="Wait for task to complete")
@click.pass_context
def release_sdk(ctx, bump, messages, client_messages, sdk_version=None, client_version=None, author=None,
                sandbox_oauth_token=None, wait=True):
    walle_sdk_project_path = os.path.join(arcadia.arcadia_root(), "infra", "wall-e", "sdk")
    walle_sdk_version_file = os.path.join(walle_sdk_project_path, "walle_api", "constants.py")
    walle_sdk_changelog_path = os.path.join(walle_sdk_project_path, "debian", "changelog")

    if bump or sdk_version:
        sdk_version = ctx.invoke(
            bump_version,
            version_file=walle_sdk_version_file,
            bump=bump,
            version=sdk_version,
            commit=False,
        )

        ctx.invoke(
            bump_changelog,
            changelog_path=walle_sdk_changelog_path,
            source_dir=walle_sdk_project_path,
            messages=messages,
            version="{}-1".format(sdk_version),
            author=author,
            commit=False,
        )

    walle_client_project_path = os.path.join(arcadia.arcadia_root(), "infra", "wall-e", "client")
    walle_client_version_file = os.path.join(walle_client_project_path, "walle_cli", "constants.py")
    walle_client_changelog_path = os.path.join(walle_client_project_path, "debian", "changelog")
    walle_client_requirements_path = os.path.join(walle_client_project_path, "requirements.txt")
    walle_client_package_path = os.path.join(walle_client_project_path, "client-deb.json")

    if sdk_version:
        logging.info("Bumping wall-e.cli dependency for wall-e.api to %s", sdk_version)
        set_minimum_dependency_version(walle_client_requirements_path, "wall-e.api", sdk_version)

    if bump or client_version:
        client_version = ctx.invoke(
            bump_version,
            version_file=walle_client_version_file,
            bump=bump,
            version=client_version,
            commit=False,
        )

        ctx.invoke(
            bump_changelog,
            changelog_path=walle_client_changelog_path,
            source_dir=walle_client_project_path,
            messages=client_messages or messages,
            version="{}-1".format(client_version),
            author=author,
            commit=False,
        )

    if sdk_version or client_version:
        logging.info("Committing changed files")
        arcadia.commit([
            walle_sdk_version_file, walle_sdk_changelog_path,
            walle_client_version_file, walle_client_changelog_path, walle_client_requirements_path
        ],
            release_message("New release of", [("wall-e.sdk", sdk_version), ("wall-e.cli", client_version)]),
            skip_check=True)

    ctx.invoke(
        release_to_pypi,
        projects=[walle_sdk_project_path, walle_client_project_path],
        message=release_message("Release", [("Wall-E.api", sdk_version), ("Wall-E.client", client_version)]),
        sandbox_oauth_token=sandbox_oauth_token,
        wait=wait,
    )

    ctx.invoke(
        release_deb,
        package=walle_client_package_path,
        message=release_message("Release", [("Wall-E.client", client_version)]),
        sandbox_oauth_token=sandbox_oauth_token,
        repo=DebRepo.COMMON,
        wait=wait,
    )


@release.command("release-client", help="Release Wall-E.cli")
@version_bump_options(with_no_bump=True)
@click.option("-a", "--author", envvar=CRD_RELEASE_AUTHOR_ENV, required=False,
              help="Author name in form 'Name Surname <name.surname@company.org>'")
@click.option("-m", "--messages", required=False, multiple=True, help="Changelog messages")
@click.option("--version", required=False, help="Version dotted string for new Client release")
@click.option("--sandbox-oauth-token", envvar=CRD_SANDBOX_OAUTH_TOKEN_ENV, required=False)
@click.option("-w/-W", "--wait/--no-wait", default=True, help="Wait for task to complete")
@click.pass_context
def release_client(ctx, bump, messages, version=None, author=None, sandbox_oauth_token=None, wait=True):

    walle_client_project_path = os.path.join(arcadia.arcadia_root(), "infra", "wall-e", "client")
    walle_client_version_file = os.path.join(walle_client_project_path, "walle_cli", "constants.py")
    walle_client_changelog_path = os.path.join(walle_client_project_path, "debian", "changelog")
    walle_client_package_path = os.path.join(walle_client_project_path, "client-deb.json")

    if bump or version:
        version = ctx.invoke(
            bump_version,
            version_file=walle_client_version_file,
            bump=bump,
            version=version,
            commit=False,
        )

        ctx.invoke(
            bump_changelog,
            changelog_path=walle_client_changelog_path,
            source_dir=walle_client_project_path,
            messages=messages,
            version="{}-1".format(version),
            author=author,
            commit=False,
        )

        logging.info("Committing changed files")
        arcadia.commit([
            walle_client_version_file, walle_client_changelog_path
        ], "New release of wall-e.cli v{}".format(version), skip_check=True)

    ctx.invoke(
        release_to_pypi,
        projects=[walle_client_project_path],
        message=release_message("Release", [("Wall-E.client", version)]),
        sandbox_oauth_token=sandbox_oauth_token,
        wait=wait,
    )

    ctx.invoke(
        release_deb,
        package=walle_client_package_path,
        message=release_message("Release", [("Wall-E.client", version)]),
        repo=DebRepo.COMMON,
        sandbox_oauth_token=sandbox_oauth_token,
        wait=wait,
    )

    ctx.invoke(
        build_to_ya_tool,
        message=release_message("Release", [("Wall-E.client", version)]),
        sandbox_oauth_token=sandbox_oauth_token,
        wait=wait,
    )


@release.command("bump-version", help="Bump package version")
@click.argument("version-file", required=True, type=click.Path(exists=True, dir_okay=False))
@version_bump_options(with_no_bump=False)
@click.option("-v", "--version", required=False, help="New version dotted string")
@click.option("-c", "--commit", required=False, default=False)
def bump_version(version_file, bump="build", version=None, commit=False):
    version = bump_version_file(version_file, version=version, bump=bump, keep_build=False)

    if commit:
        logging.info("Committing %s", version_file)
        arcadia.commit(version_file, "Version bump to {}".format(version), skip_check=True)

    return version


@release.command("bump-changelog", help="Bump debian changelog")
@click.option("-p", "--changelog-path", required=True)
@click.option("-s", "--source-dir", default=os.getcwd(), type=click.Path(exists=True, file_okay=False))
@click.option("-v", "--version", required=False, help="Version dotted string")
@click.option("-d", "--distributions", required=False, help="Debian distributions to publish to, space separated")
@click.option("-a", "--author", envvar=CRD_RELEASE_AUTHOR_ENV, required=False,
              help="Author name in form 'Name Surname <name.surname@company.org>'")
@click.option("-m", "--messages", required=False, multiple=True)
@click.option("-c", "--commit", required=False, default=False)
def bump_changelog(changelog_path, source_dir, messages, version=None, distributions=None, author=None, commit=False):
    author = get_release_author(author)

    if not messages:
        messages = arcadia.get_commit_messages_since_last_changelog(source_dir, changelog_path)

    if not messages:
        messages = ["Changelog version bump to {} by {}".format(version, author)]

    version = changelog.bump_changelog(
        changelog_path, messages,
        version=version, author=author,
        distributions=distributions)

    if commit:
        logging.info("Committing %s", changelog_path)
        arcadia.commit(changelog_path, "Changelog version bump to {}".format(version), skip_check=True)

    return version


@release.command("release-deb", help="Build deb package")
@click.argument("package", type=click.Path(exists=True, dir_okay=False))
@click.option("-m", "--message", default="Build deb package", help="Message/description for sandbox task")
@click.option("-r", "--repo", default=DebRepo.SEARCH, help="Publish package to specified debian repo")
@click.option("-t", "--sandbox-oauth-token", envvar=CRD_SANDBOX_OAUTH_TOKEN_ENV, required=False)
@click.option("-w/-W", "--wait/--no-wait", default=True, help="Wait for task to complete")
def release_deb(package, message, repo, sandbox_oauth_token=None, wait=True):
    token = get_sandbox_token(sandbox_oauth_token)

    with build_deb_bundle_task(token, arcadia.relative(package), description=message, wait=wait, repo=repo) as task_ids:
        logging.info("Task URL is {}".format(task_ids[1]))


@release.command("release-bundle", help="Build Wall-E checks juggler-bundle")
@click.option("-m", "--message", default="Build Wall-E checks bundle", help="Message/description for sandbox task")
@click.option("-t", "--sandbox-oauth-token", envvar=CRD_SANDBOX_OAUTH_TOKEN_ENV, required=False)
@click.option("-w/-W", "--wait/--no-wait", default=True, help="Wait for task to complete")
def release_bundle(message, sandbox_oauth_token=None, wait=True):
    token = get_sandbox_token(sandbox_oauth_token)

    with build_checks_bundle_task(token, description=message, wait=wait) as task_meta:
        logging.info("Task URL is {}".format(task_meta[1]))


@release.command("release-pypi", help="Build python package and upload it to pypi")
@click.argument("projects", type=click.Path(exists=True, file_okay=False), nargs=-1)
@click.option("-m", "--message", default="Release python package", help="Message/description for sandbox task")
@click.option("-t", "--sandbox-oauth-token", envvar=CRD_SANDBOX_OAUTH_TOKEN_ENV, required=False)
@click.option("-w/-W", "--wait/--no-wait", default=True, help="Wait for task to complete")
def release_to_pypi(projects, message, sandbox_oauth_token=None, wait=True):
    token = get_sandbox_token(sandbox_oauth_token)

    projects = list(map(arcadia.relative, projects))
    with release_to_pypi_task(token, projects, description=message, wait=wait) as task_meta:
        logging.info("Task URL is {}".format(task_meta[1]))


@release.command("build-yatool", help="Build Wall-E client for ya tool")
@click.option("-m", "--message", default="Build Wall-E client for ya tool", help="Message/description for sandbox task")
@click.option("-t", "--sandbox-oauth-token", envvar=CRD_SANDBOX_OAUTH_TOKEN_ENV, required=False)
@click.option("-w/-W", "--wait/--no-wait", default=True, help="Wait for task to complete")
def build_to_ya_tool(message, sandbox_oauth_token=None, wait=True):
    token = get_sandbox_token(sandbox_oauth_token)

    with build_to_ya_tool_task(token, description=message, wait=wait) as task_meta:
        logging.info("Task URL is {}".format(task_meta[1]))

    logging.info("Please MANUALLY update file arcadia/build/ya.conf.json with task id {}".format(task_meta[0]))


@release.command("release-agent", help="Release walle.agent")
@version_bump_options(with_no_bump=True)
@click.option("-a", "--author", envvar=CRD_RELEASE_AUTHOR_ENV, required=False,
              help="Author name in form 'Name Surname <name.surname@company.org>'")
@click.option("-m", "--messages", required=False, multiple=True)
@click.option("--version", required=False, help="Version dotted string for new Agent release")
@click.option("--sandbox-oauth-token", envvar=CRD_SANDBOX_OAUTH_TOKEN_ENV, required=False)
@click.option("-w/-W", "--wait/--no-wait", default=True, help="Wait for task to complete")
@click.pass_context
def release_agent(ctx, bump, messages, author=None, version=None, sandbox_oauth_token=None, wait=True):

    walle_agent_project_path = os.path.join(arcadia.arcadia_root(), "infra", "wall-e", "agent")
    walle_agent_version_file = os.path.join(walle_agent_project_path, "constants.py")
    walle_agent_changelog_path = os.path.join(walle_agent_project_path, "debian", "changelog")
    walle_agent_package_path = os.path.join(walle_agent_project_path, "agent-deb.json")

    if bump or version:
        version = ctx.invoke(
            bump_version,
            version_file=walle_agent_version_file,
            bump=bump,
            version=version,
            commit=False,
        )

        ctx.invoke(
            bump_changelog,
            changelog_path=walle_agent_changelog_path,
            source_dir=walle_agent_project_path,
            messages=messages,
            version="{}-1".format(version),
            author=author,
            commit=False,
        )

        logging.info("Committing changed files")
        arcadia.commit([
            walle_agent_version_file, walle_agent_changelog_path
        ], "New release of agent.py v{}".format(version), skip_check=True)

    ctx.invoke(
        release_deb,
        package=walle_agent_package_path,
        message=release_message("Release", [("Wall-e.agent", version)]),
        repo=DebRepo.COMMON,
        sandbox_oauth_token=sandbox_oauth_token,
        wait=wait,
    )
