import os
import logging

CRD_SANDBOX_OAUTH_TOKEN = "sandbox_oauth_token"
CRD_RELEASE_AUTHOR = "release_author"
CRD_TOKENS = {CRD_SANDBOX_OAUTH_TOKEN, CRD_RELEASE_AUTHOR}
CRD_FILENAME_CWD = ".release.rc"
CRD_FILENAME_HOME = os.path.expanduser("~/.release.rc")

CRD_SANDBOX_OAUTH_TOKEN_ENV = "SANDBOX_OAUTH_TOKEN"
CRD_RELEASE_AUTHOR_ENV = "RELEASE_AUTHOR"

SANDBOX_STATUSES_SUCCESS = {"SUCCESS", "RELEASED"}
SANDBOX_STATUSES_FAILED = {"ERROR", "FAILURE", "EXCEPTION", "TIMEOUT", "STOPPED", "NO_RES", "EXPIRED", "DELETED"}


def read_credentials_from_config_file(rc_filename):
    credentials = {}
    if not os.path.isfile(rc_filename):
        return credentials

    try:
        with open(rc_filename) as f:
            for line in f.read().splitlines():
                if not line:
                    continue

                k, v = list(map(str.strip, line.split(':', 1)))
                if k in CRD_TOKENS:
                    credentials[k] = v
    except IOError as e:
        logging.error("Failed to read credentials from %s: %s", rc_filename, e)
    except (KeyError, ValueError):
        logging.error("Incorrect RC file format [%s]", rc_filename)

    return credentials


def get_credentials_from_env():
    env_names = {CRD_SANDBOX_OAUTH_TOKEN: CRD_SANDBOX_OAUTH_TOKEN, CRD_RELEASE_AUTHOR_ENV: CRD_RELEASE_AUTHOR}
    credentials = {}

    for var in env_names:
        if var in os.environ:
            credentials[env_names[var]] = os.environ[var]

    return credentials


def get_credentials_from_args(sandbox_oauth_token=None, release_author=None):
    credentials = {}

    if sandbox_oauth_token:
        credentials[CRD_SANDBOX_OAUTH_TOKEN] = sandbox_oauth_token
    if release_author:
        credentials[CRD_RELEASE_AUTHOR] = release_author

    return credentials


def add_missing(original, other):
    for key in other.keys() - original.keys():
        original[key] = other[key]


_SKIP = object()


def get_credentials(sandbox_oauth_token=_SKIP, release_author=_SKIP):
    credentials = get_credentials_from_args(sandbox_oauth_token, release_author)

    if credentials.keys() < CRD_TOKENS:
        add_missing(credentials, get_credentials_from_env())

    if credentials.keys() < CRD_TOKENS:
        add_missing(credentials, read_credentials_from_config_file(CRD_FILENAME_CWD))

    if credentials.keys() < CRD_TOKENS:
        add_missing(credentials, read_credentials_from_config_file(CRD_FILENAME_HOME))

    if credentials.keys() < CRD_TOKENS:
        logging.error("Please, provide sandbox oauth token and release author")
        exit(2)

    return credentials


def get_sandbox_token(sandbox_oauth_token=None):
    credentials = get_credentials(sandbox_oauth_token=sandbox_oauth_token)
    return credentials[CRD_SANDBOX_OAUTH_TOKEN]


def get_release_author(release_author=None):
    credentials = get_credentials(release_author=release_author)
    return credentials[CRD_RELEASE_AUTHOR]
