import re

version_re = re.compile(r'^'
                        r'(?P<pkg_name>[\w._-]+)'
                        r'(?:(?P<min_version_op>>=?)(?P<min_version>[\da-z.]+),?)?'
                        r'(?:(?P<max_version_op><=?)(?P<max_version>[\da-z.]+))?'
                        r'(?:==(?P<strict_version>[\da-z.]+))?')


def set_minimum_dependency_version(requirements_file, dependency, version):
    requirements = read_requirements_file(requirements_file)
    write_requirements_file(requirements_file, amend_dependency(requirements, dependency, version))


def read_requirements_file(requirements_file):
    with open(requirements_file) as f:
        return [line.strip() for line in f]


def write_requirements_file(requirements_file, lines):
    with open(requirements_file, 'w') as f:
        f.write('\n'.join(lines))


def amend_dependency(requirements, dependency, version):
    for line in requirements:
        if line.lower().startswith(dependency.lower()):
            parsed = parse_requirement(line)

            if parsed['min_version'] is not None:
                parsed['min_version'] = version
            else:
                parsed['strict_version'] = version

            line = make_requirement_line(parsed)

        yield line


def parse_requirement(requirement_line):
    match = version_re.match(requirement_line)
    if not match:
        raise ValueError("unsupported requirements version spec: {}".format(requirement_line))

    return match.groupdict()


def make_requirement_line(parsed_requirement):
    res = [parsed_requirement['pkg_name']]
    if parsed_requirement['min_version'] is not None:
        res.extend([parsed_requirement['min_version_op'], parsed_requirement['min_version']])
        if parsed_requirement['max_version'] is not None:
            res.append(',')

    if parsed_requirement['max_version'] is not None:
        res.extend([parsed_requirement['max_version_op'], parsed_requirement['max_version']])

    if parsed_requirement['strict_version'] is not None:
        res.extend(['==', parsed_requirement['max_version']])

    return ''.join(res)
