import sys
from contextlib import contextmanager

from yalibrary.display import Display
from yalibrary.formatter import TermSupport, Formatter
from yalibrary.yandex.sandbox import SandboxClient, get_task_state_printer

SANDBOX_TASK_OWNER = 'WALLE'


def wait_task(client, task_id):
    client.wait_task_succeed(task_id, _create_task_printer())


def _create_task_printer():
    display = Display(sys.stdout, Formatter(TermSupport()))
    return get_task_state_printer(display)


@contextmanager
def run_task(token, task_params, wait=False):
    client = SandboxClient(token=token)
    task_id = client.create_task(**task_params)
    client.start_task(task_id)

    yield task_id, client.get_task_url(task_id)

    if wait:
        client.wait_task_succeed(task_id, _create_task_printer())


class DebRepo(object):
    SEARCH = "search"
    COMMON = "common"

    ALL = [SEARCH, COMMON]


def build_deb_bundle_task(token, package, description, wait=False, repo=DebRepo.SEARCH):
    return run_task(
        token=token,
        wait=wait,
        task_params=dict(
            owner=SANDBOX_TASK_OWNER,
            type="YA_PACKAGE",
            description=description,
            priority={
                "class": "USER",
                "subclass": "NORMAL"
            },
            context={
                "packages": package,
                "package_type": "debian",
                "publish_to": repo,
                "debian_distribution": "stable",
                "key_user": "robot-walle",
            }
        )
    )


def build_checks_bundle_task(token, description, wait=False):
    return run_task(
        token=token,
        wait=wait,
        task_params=dict(
            owner=SANDBOX_TASK_OWNER,
            type="BUILD_JUGGLER_CHECKS_BUNDLE",
            description=description,
            priority={
                "class": "USER",
                "subclass": "NORMAL"
            },
            context={
                "package_path": "infra/wall-e/checks/bundle-juggler/bundle.json",
                "bundle_name": "wall-e-checks-bundle",
                "upload_to_mds": False,
                "s3_vault_owner": "JUGGLER",
                "s3_vault_name": "juggler-s3-bundles-bucket-auth",
                "s3_endpoint": "s3.mds.yandex.net"
            }
        )
    )


def release_to_pypi_task(token, projects, description, wait=False):
    return run_task(
        token=token,
        wait=wait,
        task_params=dict(
            owner=SANDBOX_TASK_OWNER,
            type="RELEASE_PACKAGE_TO_PYPI",
            description=description,
            priority={
                "class": "USER",
                "subclass": "NORMAL"
            },
            context={
                "projects": ";".join(projects),
            }
        )
    )


def build_to_ya_tool_task(token, description, wait=False):
    return run_task(
        token=token,
        wait=wait,
        task_params=dict(
            owner=SANDBOX_TASK_OWNER,
            type="BUILD_ARCADIA_PROJECT_FOR_ALL",
            description=description,
            priority={
                "class": "USER",
                "subclass": "NORMAL"
            },
            context={
                "project": "infra/wall-e/client/bin/wall-e.exe/wall-e",
                "platforms_list": ["linux", "darwin"],
                "strip_binaries": True,
                "use_aapi_fuse": True,
                "backup_to_mds": False,
            }
        )
    )
