import logging
import re
from distutils.version import StrictVersion


class InvalidVersionFile(Exception):
    pass


class _SetVersionFailed(Exception):
    pass


def bump_version_file(version_file, version_var="version", version=None, bump='build', keep_build=True):
    with open(version_file) as f:
        content = f.read()

    prev_version = find_version(content, version_var)
    if prev_version is None:
        raise InvalidVersionFile("Can not find version var '{}' in file '{}'".format(version_var, version_file))

    if version is None:
        version = str(bump_version(prev_version, bump, keep_build))

    try:
        content = set_version(content, version_var, version)
    except _SetVersionFailed:
        raise InvalidVersionFile("Can not find version var '{}' in file '{}'".format(version_var, version_file))

    logging.info("Setting version in %s to %s", version_file, version)
    with open(version_file, 'w') as f:
        f.write(content)

    return version


def find_version(content, version_var):
    version_re = re.compile(r'''^(\s*{NAME}\s*=\s*)('[^']+'|"[^"]+")(?sm)'''.format(NAME=version_var))
    match = version_re.search(content)

    if match:
        before, version_string = match.groups()
        return version_string[1:-1]


def set_version(content, version_var, version):
    """ Replace value of version variable with new version"""
    changed = []

    def inject_version(match):
        changed.append(True)
        before, version_string = match.groups()
        quote_char = version_string[0]
        return before + quote_char + str(version) + quote_char

    version_re = r'''^(\s*{NAME}\s*=\s*)('[^']+'|"[^"]+")(?sm)'''.format(NAME=version_var)
    result = re.sub(version_re, inject_version, content)

    if not changed:
        raise _SetVersionFailed

    return result


def bump_version(prev_version, bump='build', keep_build=True):
    """
    Bump specified part of provided version string.

    :param prev_version: previous version string
    :param bump: 'major', 'minor', 'build' or 'prerelease'
    :param keep_build: do not reset build number when major or minor version bumps
    :return: new version string
    """
    version = StrictVersion(prev_version)
    major, minor, build = version.version

    if bump == 'prerelease':
        if version.prerelease is None:
            version.version = (major, minor, build + 1)
            version.prerelease = ('a', 1)
        else:
            version.prerelease = ('a', version.prerelease[1] + 1)
    else:
        if bump == 'build' and version.prerelease is None:
            version.version = (major, minor, build + 1)
        elif bump == 'minor':
            version.version = (major, minor + 1, build + 1 if keep_build else 1)
        elif bump == 'major':
            # reset minor version when bumping major version.
            version.version = (major + 1, 0, build + 1 if keep_build else 1)
        version.prerelease = None

    return version
