#!/usr/bin/env python
from __future__ import unicode_literals

import os
import time
import argparse
import calendar

import requests

from datetime import datetime
from collections import defaultdict

import matplotlib.pyplot as plt
from walle_api import WalleClient

TYPE_ADD_PROJECT = "add-project"
TYPE_CLONE_PROJECT = "clone-project"

CHART_LEN_MONTHS = 12
WALLE_ADMINS = {"{}@".format(login) for login in ("n-malakhov", "flagist", "warwish", "alexsmirnov", "dkhrutsky")}

WIKI_URL = "https://wiki-api.yandex-team.ru/_api/frontend/wall-e/project-creation-graphics/"
JING_LOGIN = "alexsmirnov"
JING_URL = "https://jing.yandex-team.ru/files/{}/".format(JING_LOGIN)
PNG_NAME = 'project-chart-{}.png'


class Client(object):
    def __init__(self, host, port=0, username=None, password=None, protocol='https', verify_ssl=True):
        if not port:
            port = 443 if protocol == 'https' else 80
        self.baseurl = '{0}://{1}:{2}'.format(protocol, host, port)
        self.cwd = '/'
        self.session = requests.session()
        self.session.verify = verify_ssl
        self.session.stream = True
        self.session.auth = (username, password)

    def upload(self, local_path_or_fileobj, remote_path):
        with open(local_path_or_fileobj, 'rb') as fileobj:
            self._send('PUT', remote_path, data=fileobj)

    def _get_url(self, path):
        path = str(path).strip()
        if path.startswith('/'):
            return self.baseurl + path
        return "".join((self.baseurl, self.cwd, path))

    def _send(self, method, path, **kwargs):
        url = self._get_url(path)
        response = self.session.request(method, url, allow_redirects=False, **kwargs)
        return response


def count_cloned_projects(client, start_time, end_time):
    return len(list(client.iter_audit_log(event_type=TYPE_CLONE_PROJECT, start_time=start_time, end_time=end_time,
                                          fields=["time"])))


def first(iterable):
    return next(iter(iterable))


def count_created_projects(client, start_time, end_time):
    admin_created, user_created = 0, 0
    for entry in client.iter_audit_log(event_type=TYPE_ADD_PROJECT, start_time=start_time, end_time=end_time,
                                       fields=["time", "issuer"]):
        if entry["issuer"] in WALLE_ADMINS:
            admin_created += 1
        # affe was in wall-e admins for some time, we don't know how to classify his entries
        elif entry["issuer"] == "affe@":
            continue
        else:
            user_created += 1
    return admin_created, user_created


class MonthDate(object):
    """Represents date without day, allows to iterate by months in both directions"""
    def __init__(self, year, month):
        self.dt = datetime(year, month, 1)

    @property
    def year(self):
        return self.dt.year

    @property
    def month(self):
        return self.dt.month

    def inc_month(self):
        year = self.year if self.month + 1 <= 12 else self.year + 1
        month = self.month + 1 if self.month + 1 <= 12 else 1
        return MonthDate(year=year, month=month)

    def dec_month(self):
        year = self.year if self.month - 1 >= 1 else self.year - 1
        month = self.month - 1 if self.month - 1 >= 1 else 12
        return MonthDate(year=year, month=month)

    def to_timestamp(self):
        return calendar.timegm(self.dt.timetuple())

    def __repr__(self):
        return "{}.{}".format(self.year, self.month)


def get_month_borders(months_back_num):
    """Returns list of pairs (datetime(1st of cur month), datetime(1st of next month))"""
    today = datetime.today()
    month_borders = []
    cur_md = MonthDate(today.year, today.month)
    for _ in range(months_back_num):
        month_borders.append((cur_md, cur_md.inc_month()))
        cur_md = cur_md.dec_month()
    return reversed(month_borders)


def get_counts_by_months(client, months_num):
    counts_by_month = defaultdict(list)
    for cur_month_md, next_month_md in get_month_borders(months_num):
        start_time, end_time = cur_month_md.to_timestamp(), next_month_md.to_timestamp()
        counts_by_month["cloned"].append(count_cloned_projects(client, start_time, end_time))
        admin, user = count_created_projects(client, start_time, end_time)
        counts_by_month["admin"].append(admin)
        counts_by_month["user"].append(user)
    return counts_by_month


def get_total_by_months(counts_by_months):
    total_by_months = defaultdict(list)
    for name, counts in counts_by_months.items():
        cur_total = 0
        for count in counts:
            cur_total += count
            total_by_months[name].append(cur_total)
    return total_by_months


def get_bar_positions(offset, bars_num):
    positions = [n + offset for n in range(bars_num)]
    return positions


def set_xticks_to_months(chart, num_months):
    chart.set_xticks(range(num_months))
    chart.set_xticklabels(map(repr, [s for s, _ in get_month_borders(num_months)]))


def draw_bar_heights(chart, rects, xpos='center'):
    """
    Attach a text label above each bar in *rects*, displaying its height.

    *xpos* indicates which side to place the text w.r.t. the center of
    the bar. It can be one of the following {'center', 'right', 'left'}.
    """

    ha = {'center': 'center', 'right': 'left', 'left': 'right'}
    offset = {'center': 0, 'right': 1, 'left': -1}

    for rect in rects:
        height = rect.get_height()
        chart.annotate('{}'.format(height),
                       xy=(rect.get_x() + rect.get_width() / 2, height),
                       xytext=(offset[xpos]*3, 3),  # use 3 points offset
                       textcoords="offset points",  # in both directions
                       ha=ha[xpos], va='bottom')


def draw_bar_chart(chart, counts_by_months):
    for offset, (name, counts) in zip((-0.3, 0, 0.3), counts_by_months.items()):
        bar_positions = get_bar_positions(offset, len(counts))
        rects = chart.bar(bar_positions, counts, label=name, align="center", width=0.3)
        draw_bar_heights(chart, rects)

    chart.legend()


def draw_aggr_chart(chart, counts_by_months):
    total_by_months = get_total_by_months(counts_by_months)
    for name, totals in total_by_months.items():
        chart.plot(range(len(totals)), totals, label=name)

    chart.legend()


def draw_chart(counts_by_months):
    plt.rcParams["figure.figsize"] = [20, 10]
    _, (aggr_chart, bar_chart) = plt.subplots(2)

    draw_aggr_chart(aggr_chart, counts_by_months)
    draw_bar_chart(bar_chart, counts_by_months)

    months_num = len(first(counts_by_months.values()))
    # Yes, we cannot do it in draw_*_chart, all hail matplotlib
    set_xticks_to_months(aggr_chart, months_num)
    set_xticks_to_months(bar_chart, months_num)

    filename = PNG_NAME.format(time.time())
    plt.savefig(filename)

    return filename, months_num


TEMPLATE = """

===Что это===
На этой странице находятся графики, показывающие динамику создания проектов.

{}
"""


def upload_to_jing(filename, path):
    webdav = Client('jingdav.yandex-team.ru', username='jing', password='jing', port=443, protocol="https")
    webdav.upload(path, '{}/{}'.format(JING_LOGIN, filename))


def push_to_wiki(filename, token, first_month, last_month):
    data = {"title": "Графики, показывающие количество созданных проектов c {} по {}".format(first_month, last_month),
            "body": TEMPLATE.format(JING_URL + filename)}
    requests.post(WIKI_URL, headers={"Authorization": "OAuth {}".format(token)}, json=data)


def get_border_months(months_num):
    months = list(get_month_borders(months_num))
    first_month, last_month = months[0][0], months[-1][0]
    return first_month, last_month


def _parse_args():
    parser = argparse.ArgumentParser(description="Draw charts of projects created by users.")
    parser.add_argument("months", default=CHART_LEN_MONTHS, nargs="?", help="Draw a chart for MONTHS months")
    parser.add_argument("--walle-token", required=True, help="Your wall-e OAuth token")
    parser.add_argument("--save-png", action='store_true', help="Don't remove result png")
    return parser.parse_args()


def main():
    args = _parse_args()
    client = WalleClient()

    counts_by_months = get_counts_by_months(client, args.months)
    filename, months_num = draw_chart(counts_by_months)
    first_month, last_month = get_border_months(months_num)

    path_to_file = os.path.join(os.path.dirname(os.path.abspath(__file__)), filename)
    upload_to_jing(path_to_file, filename)

    push_to_wiki(filename, args.walle_token, first_month, last_month)

    if not args.save_png and os.path.isfile(filename):
        os.remove(filename)


if __name__ == "__main__":
    main()
