"""Wall-E API client"""

from __future__ import unicode_literals

import cgi
import sys
from collections import OrderedDict

import requests
import requests.adapters
import urllib3.util.retry


try:
    import json
except ImportError:
    import simplejson as json
from requests.exceptions import RequestException

from . import constants


class _Error(Exception):
    def __init__(self, *args, **kwargs):
        message, args = args[0], args[1:]
        super(_Error, self).__init__(message.format(*args, **kwargs) if args or kwargs else message)


class WalleClientError(_Error):
    pass


class WalleConnectionError(WalleClientError):
    pass


class WalleAuthorizationError(WalleClientError):
    pass


class WalleApiError(WalleClientError):
    def __init__(self, status_code, message):
        super(WalleApiError, self).__init__(message)
        self.status_code = status_code


class WalleClient(object):
    MAX_PAGE_SIZE = 1000
    __api_action_http_methods = {"add": "POST", "remove": "DELETE", "set": "PUT"}

    def __init__(self, url=None, access_token=None, name=None):
        if url is None:
            url = constants.PRODUCTION_URL

        self.__api = _Api(url, name=name, access_token=access_token)

    # projects api
    def add_project(self, id, name, owners=None, default_host_restrictions=None,
                    provisioner=None, deploy_config=None, deploy_tags=None, deploy_network=None,
                    certificate_deploy=False, dns_domain=None, host_shortname_template=None,
                    hbf_project_id=None, ip_method=None, bot_project_id=None, tags=None, reboot_via_ssh=None,
                    reason=None):

        return self.__api.post("/projects", request=_drop_none({
            "id": id,
            "name": name,
            "tags": tags or None,
            "owners": owners,
            "bot_project_id": bot_project_id,
            "dns_domain": dns_domain,
            "host_shortname_template": host_shortname_template,
            "hbf_project_id": hbf_project_id,
            "ip_method": ip_method,
            "provisioner": provisioner,
            "deploy_config": deploy_config,
            "deploy_tags": deploy_tags,
            "deploy_network": deploy_network,
            "certificate_deploy": certificate_deploy,
            "default_host_restrictions": default_host_restrictions,
            "reboot_via_ssh": reboot_via_ssh,
            "reason": reason
        }))

    def set_project_cms(self, project_id, cms_settings, reason=None):
        return self.__api.post("/projects/{}/cms_settings".format(project_id),
                               request=_drop_none(dict(cms_settings=cms_settings,
                                                       reason=reason)))

    def get_project(self, id, fields=None):
        return self.__api.get("/projects/" + id, params={"fields": _fields(fields)})

    def get_projects(self, fields=None, tags=None):
        return self.__api.get("/projects", params={"fields": _fields(fields), "tags": tags})

    def clone_project(self, orig_project_id, id, name, reason=None):
        return self.__api.post("/projects/clone/{}".format(orig_project_id), request={
            "id": id, "name": name, "reason": reason
        })

    def modify_project(self, id, name=None, hbf_project_id=None, ip_method=None, bot_project_id=None,
                       default_host_restrictions=None, certificate_deploy=None, tags=None,
                       host_shortname_template=None, reason=None):
        request = {
            "name": name, "tags": tags or None, "certificate_deploy": certificate_deploy,
            "hbf_project_id": hbf_project_id, "ip_method": ip_method, "bot_project_id": bot_project_id,
            "host_shortname_template": host_shortname_template,
            "default_host_restrictions": default_host_restrictions, "reason": reason}

        return self.__api.post("/projects/" + id, request=request)

    def modify_project_owners(self, id, action, owners, reason=None):
        return self.__action(action, "/projects/" + id + "/owners", request={"owners": owners, "reason": reason})

    def modify_project_owned_vlans(self, id, action, vlans, reason=None):
        return self.__action(action, "/projects/" + id + "/owned_vlans", request={"vlans": vlans, "reason": reason})

    def set_profiling_config(self, id, profile, profile_tags=None, reason=None):
        return self.__api.put("/projects/" + id + "/host-profiling-config", request={
            "profile": profile, "profile_tags": profile_tags, "reason": reason})

    def set_provisioner_config(self, id, provisioner, deploy_config,
                               deploy_config_policy=None, deploy_tags=None, deploy_network=None,
                               reason=None):
        return self.__api.put("/projects/" + id + "/host-provisioner-config", request={
            "provisioner": provisioner,
            "deploy_config": deploy_config,
            "deploy_config_policy": deploy_config_policy,
            "deploy_tags": deploy_tags,
            "deploy_network": deploy_network,
            "reason": reason
        })

    def set_project_vlan_scheme(self, id, scheme, native_vlan, extra_vlans=None, reason=None):
        return self.__api.put("/projects/" + id + "/vlan_scheme", request={
            "scheme": scheme, "native_vlan": native_vlan, "extra_vlans": extra_vlans, "reason": reason})

    def reset_project_vlan_scheme(self, id, reason=None):
        return self.__api.delete("/projects/" + id + "/vlan_scheme", request={"reason": reason})

    def set_project_hbf_project_id(self, id, hbf_project_id, ip_method=None, reason=None):
        return self.__api.post("/projects/" + id + "/hbf_project_id",
                               request={"hbf_project_id": hbf_project_id, "ip_method": ip_method, "reason": reason})

    def unset_project_hbf_project_id(self, id, reason=None):
        return self.__api.delete("/projects/" + id + "/hbf_project_id", request={"reason": reason})

    def set_project_bot_project_id(self, id, bot_project_id, reason=None):
        return self.__api.post("/projects/" + id + "/bot_project_id",
                               request={"bot_project_id": bot_project_id, "reason": reason})

    def unset_project_bot_project_id(self, id, reason=None):
        return self.__api.delete("/projects/" + id + "/bot_project_id", request={"reason": reason})

    def set_project_dns_domain(self, id, dns_domain, reason=None):
        return self.__api.post("/projects/" + id + "/dns_domain",
                               request={"dns_domain": dns_domain, "reason": reason})

    def set_project_repair_request_severity(self, id, severity, reason=None):
        return self.__api.post("/projects/" + id + "/repair_request_severity",
                               request={"repair_request_severity": severity, "reason": reason})

    def unset_project_dns_domain(self, id, reason=None):
        return self.__api.delete("/projects/" + id + "/dns_domain", request={"reason": reason})

    def set_project_host_shortname_template(self, id, host_shortname_template, reason=None):
        return self.__api.post("/projects/" + id + "/host_shortname_template",
                               request={"host_shortname_template": host_shortname_template, "reason": reason})

    def unset_project_host_shortname_template(self, id, reason=None):
        return self.__api.delete("/projects/" + id + "/host_shortname_template", request={"reason": reason})

    def modify_project_tags(self, id, action, tags, reason=None):
        return self.__action(action, "/projects/" + id + "/tags", request={"tags": tags, "reason": reason})

    def modify_project_enable_reboot_via_ssh(self, id, reason=None):
        return self.__api.put("/projects/" + id + "/rebooting_via_ssh", request={"reason": reason})

    def modify_project_disable_reboot_via_ssh(self, id, reason=None):
        return self.__api.delete("/projects/" + id + "/rebooting_via_ssh", request={"reason": reason})

    def set_project_automation_plot(self, id, automation_plot_id, reason=None):
        return self.__api.post(
            "/projects/" + id + "/automation_plot",
            request={"automation_plot_id": automation_plot_id, "reason": reason}
        )

    def unset_project_automation_plot(self, id, reason):
        return self.__api.delete("/projects/" + id + "/automation_plot", request={"reason": reason})

    def enable_project_fsm_handbrake(self, id, timeout=None, timeout_time=None, ticket_key=None, reason=None):
        return self.__api.post(
            "/projects/" + id + "/fsm-handbrake",
            request={"timeout": timeout, "timeout_time": timeout_time, "ticket_key": ticket_key, "reason": reason}
        )

    def disable_project_fsm_handbrake(self, id, reason=None):
        return self.__api.delete("/projects/" + id + "/fsm-handbrake", request={"reason": reason})

    def enable_automation(self, id, automation_type=constants.AutomationType.TYPE_ALL_AUTOMATION,
                          credit_time=None, reason=None,
                          max_unreachable_failures=None, max_ssh_failures=None,
                          max_memory_failures=None, max_disk_failures=None, max_link_failures=None,
                          max_cpu_failures=None, max_overheat_failures=None, max_bmc_failures=None,
                          max_reboots_failures=None, max_tainted_kernel_failures=None,
                          max_cpu_capping_failures=None, max_fs_check_failures=None, max_checks_missing_failures=None,
                          max_dead_hosts=None, max_dns_fixes=None, **other_limits
                          ):

        credit = dict(
            max_unreachable_failures=max_unreachable_failures,
            max_ssh_failures=max_ssh_failures,
            max_memory_failures=max_memory_failures,
            max_disk_failures=max_disk_failures,
            max_link_failures=max_link_failures,
            max_cpu_failures=max_cpu_failures,
            max_overheat_failures=max_overheat_failures,
            max_bmc_failures=max_bmc_failures,
            max_reboots_failures=max_reboots_failures,
            max_tainted_kernel_failures=max_tainted_kernel_failures,
            max_cpu_capping_failures=max_cpu_capping_failures,
            max_fs_check_failures=max_fs_check_failures,
            max_checks_missing_failures=max_checks_missing_failures,
            max_dead_hosts=max_dead_hosts,
            max_dns_fixes=max_dns_fixes,
            **other_limits
        )

        if automation_type != constants.AutomationType.TYPE_HEALING_AUTOMATION:
            credit = _filter_dict_keys(credit, constants.AutomationType.AUTOMATION_TYPE_LIMIT_MAP[automation_type])

        automation_type_path = constants.AutomationType.AUTOMATION_TYPE_PATH_MAP[automation_type]
        return self.__api.put("/projects/" + id + "/enable_automation" + automation_type_path, request={
            "credit": _drop_none(dict(credit, time=credit_time)) or None,
            "reason": reason,
        })

    def disable_automation(self, id, automation_type=constants.AutomationType.TYPE_ALL_AUTOMATION, reason=None):
        automation_type_path = constants.AutomationType.AUTOMATION_TYPE_PATH_MAP[automation_type]
        return self.__api.delete("/projects/" + id + "/enable_automation" + automation_type_path,
                                 request={"reason": reason})

    def modify_project_automation_limits(self, id, reason=None,
                                         max_unreachable_failures=None, max_ssh_failures=None,
                                         max_memory_failures=None, max_disk_failures=None, max_link_failures=None,
                                         max_cpu_failures=None, max_overheat_failures=None, max_bmc_failures=None,
                                         max_reboots_failures=None, max_tainted_kernel_failures=None,
                                         max_cpu_capping_failures=None, max_fs_check_failures=None,
                                         max_checks_missing_failures=None, max_dead_hosts=None, max_dns_fixes=None,
                                         **other_limits
                                         ):

        self.__api.post("/projects/" + id + "/automation_limits", request=dict(
            reason=reason,
            max_unreachable_failures=max_unreachable_failures,
            max_ssh_failures=max_ssh_failures,
            max_memory_failures=max_memory_failures,
            max_disk_failures=max_disk_failures,
            max_link_failures=max_link_failures,
            max_cpu_failures=max_cpu_failures,
            max_overheat_failures=max_overheat_failures,
            max_bmc_failures=max_bmc_failures,
            max_reboots_failures=max_reboots_failures,
            max_tainted_kernel_failures=max_tainted_kernel_failures,
            max_cpu_capping_failures=max_cpu_capping_failures,
            max_fs_check_failures=max_fs_check_failures,
            max_checks_missing_failures=max_checks_missing_failures,
            max_dead_hosts=max_dead_hosts,
            max_dns_fixes=max_dns_fixes,
            **other_limits
        ))

    def modify_project_notification_recipients(self, id, action, reason=None, audit=None, info=None, warning=None,
                                               bot=None, error=None, critical=None):
        return self.__action(action, "/projects/" + id + "/notifications/recipients", request={
            "audit": audit, "info": info, "warning": warning, "bot": bot, "error": error, "critical": critical,
            "reason": reason})

    def set_failure_report_parameters(self, id, enabled=None, queue=None, summary=None, extra=None, reason=None):
        return self.__api.put("/projects/" + id + "/reports", request={
            "enabled": enabled, "queue": queue, "summary": summary, "extra": extra, "reason": reason
        })

    def modify_failure_report_parameters(self, id, enabled=None, queue=None, summary=None, extra=None, reason=None):
        return self.__api.patch("/projects/" + id + "/reports", request={
            "enabled": enabled, "queue": queue, "summary": summary, "extra": extra, "reason": reason
        })

    def remove_failure_report_parameters(self, id, reason=None):
        return self.__api.delete("/projects/" + id + "/reports", request={"reason": reason})

    def remove_project(self, id, reason=None):
        return self.__api.delete("/projects/" + id, request={"reason": reason})

    # automation plot api
    def add_automation_plot(self, id, name, owners, checks=None, reason=None):
        return self.__api.post(
            "/automation-plot/",
            request={
                "id": id, "name": name, "owners": owners, "checks": checks, "reason": reason
            })

    def get_automation_plot(self, id, fields=None):
        return self.__api.get("/automation-plot/" + id, params={"fields": _fields(fields)})

    def get_automation_plots(self, fields=None):
        return self.__api.get("/automation-plot/", params={"fields": _fields(fields)})

    def modify_automation_plot(self, id, name=None, owners=None, checks=None, reason=None):
        return self.__api.patch(
            "/automation-plot/" + id,
            request={"name": name, "owners": owners, "checks": checks, "reason": reason}
        )

    def enable_check(self, id, check_name, reason=None):
        return self.__api.post(
            "/automation-plot/" + id + "/" + check_name + "/enable",
            request={"reason": reason}
        )

    def disable_check(self, id, check_name, reason=None):
        return self.__api.post(
            "/automation-plot/" + id + "/" + check_name + "/disable",
            request={"reason": reason}
        )

    def remove_automation_plot(self, id, reason):
        return self.__api.delete("/automation-plot/" + id, request={"reason": reason})

    # preorder api
    def add_preorder(self, id, project, prepare=False, provisioner=None, deploy_config=None, restrictions=None,
                     sudo=None, reason=None):
        return self.__api.post("/preorders", params={"sudo": sudo}, request={
            "id": id, "project": project, "prepare": prepare, "provisioner": provisioner,
            "deploy_config": deploy_config, "restrictions": restrictions, "reason": reason})

    def get_preorder(self, id, fields=None):
        return self.__api.get("/preorders/{}".format(id), params={"fields": _fields(fields)})

    def get_preorders(self, fields=None):
        return self.__api.get("/preorders", params={"fields": _fields(fields)})

    def restart_preorder(self, id, sudo=None, reason=None):
        return self.__api.post("/preorders/{}/restart".format(id), params={"sudo": sudo}, request={"reason": reason})

    def remove_preorder(self, id, sudo=None, reason=None):
        return self.__api.delete("/preorders/{}".format(id), params={"sudo": sudo}, request={"reason": reason})

    # host api
    def get_host(self, host_id, resolve_deploy_configuration=None, fields=None, resolve_tags=None):
        host_fields, network_fields = self.__get_host_network_fields(fields)
        params=_drop_none({
            "resolve_deploy_configuration": resolve_deploy_configuration,
            "fields": _fields(host_fields),
            "resolve_tags": resolve_tags,
        })
        host = self.__api.get(self.__host_uri(host_id), params=params)
        if network_fields:
            host_network = self.get_host_network(host["uuid"], network_fields)
            for field in network_fields:
                if field in host_network:
                    host[field] = host_network.get(field, None)
        return host

    def get_host_configuration(self, host_id):
        return self.__api.get(self.__host_uri(host_id, "/current-configuration"))

    def iter_hosts(self, fields=None, limit=None, **kwargs):
        if "offset" in kwargs:
            raise TypeError("Host iterator doesn't support offset.")

        cursor = 0
        host_num = 0
        drop_cursor_field = False

        if fields is not None and "inv" not in fields:
            fields = list(fields) + ["inv"]
            drop_cursor_field = True

        while limit is None or host_num < limit:
            if limit is None:
                request_limit = self.MAX_PAGE_SIZE
            else:
                request_limit = min(limit - host_num, self.MAX_PAGE_SIZE)

            result = self.get_hosts(fields=fields, cursor=cursor, limit=request_limit, **kwargs)
            hosts = result["result"]
            if not hosts:
                break

            for host in hosts:
                if drop_cursor_field:
                    del host["inv"]
                yield host

            if "next_cursor" in result:
                cursor = result["next_cursor"]
            else:
                break

            host_num += len(hosts)

    def get_hosts(self, names=None, invs=None, uuids=None, name=None, state=None, status=None, health=None,
                  project=None, tags=None, provisioner=None, config=None, deploy_config_policy=None,
                  restrictions=None, task_owner=None,
                  physical_location=None, switch=None, port=None, resolve_deploy_configuration=False, scenario_id=None,
                  fields=None, cursor=None, offset=None, limit=None):

        host_fields, network_fields = self.__get_host_network_fields(fields)
        resolve_tags = True if fields and "tags" in host_fields else None

        hosts_params = {
            "name": name,
            "state": state,
            "status": status,
            "health": health,
            "project": project,
            "tags": tags,
            "task_owner": task_owner,
            "provisioner": provisioner,
            "config": config,
            "deploy_config_policy": deploy_config_policy,
            "restrictions": restrictions,
            "physical_location": physical_location,
            "switch": switch,
            "port": port,
            "scenario_id": scenario_id,
            "resolve_deploy_configuration": resolve_deploy_configuration,
            "resolve_tags": resolve_tags,
            "fields": _fields(host_fields),
            "cursor": cursor,
            "offset": offset,
            "limit": limit,
        }

        if invs is None and names is None and uuids is None:
            response = self.__api.get("/hosts", params=hosts_params)
        else:
            response = self.__api.post("/get-hosts",
                                       request={"invs": invs, "names": names, "uuids": uuids}, params=hosts_params)
        hosts = response["result"]

        if network_fields:
            uuid_host_map = {host["uuid"]: host for host in hosts}
            for host_network in self.get_hosts_network(list(uuid_host_map.keys()),
                                                       list(network_fields), limit=limit)["result"]:
                for field in network_fields:
                    if field in host_network:
                        uuid_host_map[host_network["uuid"]][field] = host_network.get(field, None)

        return response

    def add_host(self, host_id, project, provisioner=None, config=None, deploy_config_policy=None,
                 restrictions=None, state=None, deploy_tags=None, deploy_network=None, status=None,
                 ignore_cms=None, disable_admin_requests=None, check=None, with_auto_healing=None,
                 dns=None, instant=None, reason=None, maintenance_params=None):
        if is_number(host_id):
            inv, name = host_id, None
        else:
            inv, name = None, host_id

        return self.__api.post("/hosts", request=_drop_none({
            "inv": inv,
            "name": name,
            "project": project,
            "provisioner": provisioner,
            "config": config,
            "deploy_config_policy": deploy_config_policy,
            "deploy_tags": deploy_tags,
            "deploy_network": deploy_network,
            "restrictions": restrictions,
            "state": state,
            "status": status,
            "reason": reason,
            "ignore_cms": ignore_cms,
            "dns": dns,
            "instant": instant,
            "disable_admin_requests": disable_admin_requests,
            "check": check,
            "with_auto_healing": with_auto_healing,
            "maintenance_properties": maintenance_params
        }))

    def modify_host(self, host_id, restrictions=None, ignore_maintenance=None, reason=None):
        return self.__api.post(self.__host_uri(host_id), request={"restrictions": restrictions, "reason": reason},
                               params={"ignore_maintenance": ignore_maintenance})

    def modify_host_extra_vlans(self, host_id, action, vlans, ignore_maintenance=None, reason=None):
        return self.__action(action, self.__host_uri(host_id, "/extra_vlans"),
                             params={"ignore_maintenance": ignore_maintenance},
                             request={"vlans": vlans, "reason": reason})

    def set_deploy_config(self, host_id, config_name, provisioner=None, deploy_tags=None, deploy_config_policy=None,
                          deploy_network=None, certificate_deploy=None, ignore_maintenance=None, reason=None):
        request = {"config": config_name,
                   "provisioner": provisioner,
                   "deploy_config_policy": deploy_config_policy,
                   "deploy_tags": deploy_tags,
                   "deploy_network": deploy_network,
                   "certificate_deploy": certificate_deploy,
                   "reason": reason}

        return self.__api.put(self.__host_uri(host_id, "/deploy_config"), request=request,
                              params={"ignore_maintenance": ignore_maintenance})

    def clear_deploy_config(self, host_id, ignore_maintenance=None, reason=None):
        return self.__api.delete(self.__host_uri(host_id, "/deploy_config"), request={"reason": reason},
                                 params={"ignore_maintenance": ignore_maintenance})

    def set_host_maintenance(self, host_id, ticket_key=None, power_off=None, timeout_time=None, timeout_status=None,
                             ignore_cms=None, disable_admin_requests=None, cms_task_action=None, operation_state=None,
                             reason=None):
        return self.__api.post(self.__host_uri(host_id, "/set-maintenance"),
                               request={"power_off": power_off, "timeout_time": timeout_time,
                                        "timeout_status": timeout_status, "ticket_key": ticket_key,
                                        "ignore_cms": ignore_cms, "disable_admin_requests": disable_admin_requests,
                                        "cms_task_action": cms_task_action, "operation_state": operation_state,
                                        "reason": reason}
                               )

    def change_host_maintenance(self, host_id, ticket_key=None, timeout_time=None, timeout_status=None,
                                operation_state=None, reason=None, remove_timeout=None):

        return self.__api.post(self.__host_uri(host_id, "/change-maintenance"),
                               request=_drop_none({
                                   "timeout_time": timeout_time, "timeout_status": timeout_status,
                                   "ticket_key": ticket_key, "operation_state": operation_state,
                                   "reason": reason, "remove_timeout": remove_timeout}))

    def set_host_assigned(self, host_id, status=None, power_on=None, ignore_maintenance=None,
                          disable_admin_requests=None, check=None, auto_healing=None, reason=None):
        return self.__api.post(self.__host_uri(host_id, "/set-assigned"),
                               params={"ignore_maintenance": ignore_maintenance},
                               request={"status": status, "power_on": power_on,
                                        "check": check, "auto_healing": auto_healing,
                                        "disable_admin_requests": disable_admin_requests,
                                        "reason": reason})

    def force_host_status(self, host_id, status, timeout_time=None, timeout_status=None,
                          ignore_maintenance=None, reason=None, ticket_key=None):
        return self.__api.post(self.__host_uri(host_id, "/force-status"),
                               params={"ignore_maintenance": ignore_maintenance},
                               request={"status": status, "timeout_time": timeout_time,
                                        "timeout_status": timeout_status, "reason": reason,
                                        "ticket_key": ticket_key})

    def cancel_host_task(self, host_id, ignore_maintenance=False, reason=None):
        return self.__api.post(self.__host_uri(host_id, "/cancel-task"),
                               params={"ignore_maintenance": ignore_maintenance},
                               request={"reason": reason})

    def switch_vlans(self, host_id, vlans=None, native_vlan=None, network_target=None,
                     ignore_maintenance=None, reason=None):
        return self.__api.post(self.__host_uri(host_id, "/switch-vlans"), request={
            "vlans": vlans, "native_vlan": native_vlan, "network_target": network_target, "reason": reason
        }, params={"ignore_maintenance": ignore_maintenance})

    def switch_project(self, host_id, project_id, release=False, erase_disks=None, restrictions=None, force=None,
                       ignore_cms=None, disable_admin_requests=None, ignore_maintenance=None, reason=None):
        return self.__api.post(self.__host_uri(host_id, "/switch-project"), request={
            "project": project_id, "release": release, "erase_disks": erase_disks, "restrictions": restrictions,
            "force": force, "ignore_cms": ignore_cms, "disable_admin_requests": disable_admin_requests, "reason": reason
        }, params={"ignore_maintenance": ignore_maintenance})

    def release_host(self, host_id, erase_disks=None, ignore_cms=None, disable_admin_requests=None,
                     ignore_maintenance=None, reason=None):
        return self.__api.post(self.__host_uri(host_id, "/release-host"), request={
            "erase_disks": erase_disks, "ignore_cms": ignore_cms,
            "disable_admin_requests": disable_admin_requests, "reason": reason
        }, params={"ignore_maintenance": ignore_maintenance})

    def power_on_host(self, host_id, disable_admin_requests=None, check=None, with_auto_healing=None,
                      ignore_maintenance=None, reason=None):
        return self.__api.post(self.__host_uri(host_id, "/power-on"), request={
            "disable_admin_requests": disable_admin_requests, "check": check, "with_auto_healing": with_auto_healing,
            "reason": reason,
        }, params={"ignore_maintenance": ignore_maintenance})

    def power_off_host(self, host_id, ignore_cms=None, disable_admin_requests=None,
                       ignore_maintenance=None, operation_state=None, reason=None,
                       ticket_key=None, timeout_status=None, timeout_time=None, cms_task_action=None):
        return self.__api.post(self.__host_uri(host_id, "/power-off"), request={
            "ignore_cms": ignore_cms,
            "disable_admin_requests": disable_admin_requests,
            "operation_state": operation_state,
            "reason": reason,
            "ticket_key": ticket_key,
            "timeout_status": timeout_status,
            "timeout_time": timeout_time,
            "cms_task_action": cms_task_action,
        }, params={"ignore_maintenance": ignore_maintenance})

    def reboot_host(self, host_id, ssh=None, ignore_cms=None, disable_admin_requests=None, check=None,
                    with_auto_healing=None, ignore_maintenance=None, reason=None):
        return self.__api.post(self.__host_uri(host_id, "/reboot"), request={
            "ssh": ssh, "ignore_cms": ignore_cms, "disable_admin_requests": disable_admin_requests, "check": check,
            "with_auto_healing": with_auto_healing, "reason": reason
        }, params={"ignore_maintenance": ignore_maintenance})

    def check_host_dns(self, host_id, disable_admin_requests=None, check=None, with_auto_healing=None,
                       ignore_maintenance=None, reason=None):
        return self.__api.post(self.__host_uri(host_id, "/check-dns"), request={
            "disable_admin_requests": disable_admin_requests,
            "check": check, "with_auto_healing": with_auto_healing, "reason": reason,
        }, params={"ignore_maintenance": ignore_maintenance})

    def profile_host(self, host_id, profile=None, profile_tags=None, redeploy=None, provisioner=None,
                     config=None, deploy_tags=None, deploy_network=None, deploy_config_policy=None,
                     ignore_cms=None, disable_admin_requests=None,
                     check=None, with_auto_healing=None, ignore_maintenance=None, reason=None):
        return self.__api.post(self.__host_uri(host_id, "/profile"), request={
            "profile": profile,
            "profile_tags": profile_tags,
            "redeploy": redeploy,
            "provisioner": provisioner,
            "config": config,
            "deploy_config_policy": deploy_config_policy,
            "deploy_tags": deploy_tags,
            "deploy_network": deploy_network,
            "ignore_cms": ignore_cms,
            "disable_admin_requests": disable_admin_requests,
            "check": check,
            "with_auto_healing": with_auto_healing,
            "reason": reason,
        }, params={"ignore_maintenance": ignore_maintenance})

    def redeploy_host(self, host_id, config=None, provisioner=None, ignore_cms=None, disable_admin_requests=None,
                      deploy_tags=None, deploy_network=None, deploy_config_policy=None,
                      check=None, with_auto_healing=None,
                      ignore_maintenance=None, reason=None):
        return self.__api.post(self.__host_uri(host_id, "/redeploy"), request={
            "config": config,
            "ignore_cms": ignore_cms,
            "disable_admin_requests": disable_admin_requests,
            "check": check,
            "with_auto_healing": with_auto_healing,
            "reason": reason,
            "provisioner": provisioner,
            "tags": deploy_tags,
            "network": deploy_network,
            "deploy_config_policy": deploy_config_policy,
        }, params={"ignore_maintenance": ignore_maintenance})

    def prepare_host(self, host_id, config=None, profile=None, skip_profile=None, provisioner=None, restrictions=None,
                     profile_tags=None, deploy_tags=None, deploy_network=None, deploy_config_policy=None,
                     ignore_cms=None, disable_admin_requests=None, check=None, with_auto_healing=None,
                     ignore_maintenance=None, keep_name=None, reason=None):
        return self.__api.post(self.__host_uri(host_id, "/prepare"), request={
            "profile": profile,
            "provisioner": provisioner,
            "config": config,
            "restrictions": restrictions,
            "disable_admin_requests": disable_admin_requests,
            "ignore_cms": ignore_cms,
            "check": check,
            "with_auto_healing": with_auto_healing,
            "reason": reason,
            "skip_profile": skip_profile,
            "profile_tags": profile_tags,
            "deploy_tags": deploy_tags,
            "deploy_network": deploy_network,
            "deploy_config_policy": deploy_config_policy,
            "keep_name": keep_name
        }, params={"ignore_maintenance": ignore_maintenance})

    def remove_host(self, host_id, lui=None, ignore_maintenance=None,
                    ignore_cms=None, instant=None, disable_admin_requests=None, reason=None):
        return self.__api.delete(self.__host_uri(host_id), request={"reason": reason},
                                 params={"lui": lui, "ignore_maintenance": ignore_maintenance,
                                         "ignore_cms": ignore_cms, "instant": instant,
                                         "disable_admin_requests": disable_admin_requests})

    def fqdn_deinvalidation(self, host_id, release=None, clear_old_fqdn_records=None, ignore_maintenance=None,
                            ignore_cms=None, disable_admin_requests=None, reason=None):
        return self.__api.post(self.__host_uri(host_id, "/fqdn-deinvalidation"),
                               request={"reason": reason, "release": release,
                                        "clear_old_fqdn_records": clear_old_fqdn_records,
                                        "ignore_cms": ignore_cms,
                                        "disable_admin_requests": disable_admin_requests},
                               params={"ignore_maintenance": ignore_maintenance})

    def handle_host_failure(self, host_id, ignore_cms=None, ignore_maintenance=None, reason=None):
        return self.__api.post(self.__host_uri(host_id, "/handle-failure"),
                               request={"reason": reason, "ignore_cms": ignore_cms},
                               params={"ignore_maintenance": ignore_maintenance})

    def iter_audit_log(self, fields, reverse=False, limit=None, **kwargs):
        get_log_func = self._get_audit_log
        return self._get_log(get_log_func, fields, reverse, limit, **kwargs)

    def iter_operation_log(self, fields, reverse=False, limit=None, **kwargs):
        get_log_func = self._get_operation_log
        return self._get_log(get_log_func, fields, reverse, limit, **kwargs)

    def _get_log(self, get_log_func, fields, reverse=False, limit=None, **kwargs):
        # Audit log API uses entry time for cursor value. It's possible that a few entries have the same time, so we
        # have to workaround this by looking at entry IDs which are unique in contrast to entry time.
        # Also we must be very careful with floating point arithmetic.

        ids_by_time = OrderedDict()

        def on_entry(entry):
            """Returns true if we've already yielded this entry."""

            entry_id = entry["id"]
            entry_time = int(entry["time"])

            ids = ids_by_time.get(entry_time)

            if ids is None:
                ids_by_time[entry_time] = [entry_id]
                if len(ids_by_time) > 2:
                    ids_by_time.popitem(last=False)

                return False

            if entry_id in ids:
                return True

            ids.append(entry_id)
            return False

        fields = list(fields)
        drop_cursor_fields = []

        for cursor_field in "id", "time":
            if cursor_field not in fields:
                fields.append(cursor_field)
                drop_cursor_fields.append(cursor_field)

        entry_num = 0
        cursor = None

        while limit is None or entry_num < limit:
            if limit is None:
                request_limit = self.MAX_PAGE_SIZE
            else:
                request_limit = min(limit - entry_num, self.MAX_PAGE_SIZE)

            result = get_log_func(fields=fields, reverse=reverse, cursor=cursor, limit=request_limit, **kwargs)
            entries = result["result"]
            if not entries:
                break

            for entry in entries:
                if on_entry(entry):
                    continue

                for cursor_field in drop_cursor_fields:
                    del entry[cursor_field]

                entry_num += 1
                yield entry

            if "next_cursor" in result:
                next_cursor = result["next_cursor"]
            else:
                break

            # Be very careful with floating point arithmetic here
            if cursor is not None and (next_cursor >= cursor if reverse else next_cursor <= cursor):
                raise WalleClientError("Can't iterate over log: too many results for one cursor ID ({}). "
                                       "Please contact the developers.", cursor)

            cursor = next_cursor

    def get_audit_log_entry(self, entry_id, fields=None):
        return self.__api.get("/audit-log/" + entry_id, params={"fields": _fields(fields)})

    def get_operation_log_entry(self, entry_id, fields=None):
        return self.__api.get("/operation-log/" + entry_id, params={"fields": _fields(fields)})

    def _get_operation_log(self, id=None, audit_log_id=None, inv=None, name=None, scenario_id=None, type=None,
                           start_time=None, end_time=None, reverse=True, fields=None, cursor=None, limit=None):
        return self.__api.get("/operation-log", params={
            "id": id, "audit_log_id": audit_log_id, "host_inv": inv, "host_name": name, "type": type,
            "scenario_id": scenario_id, "start_time": _float(start_time), "end_time": _float(end_time),
            "fields": _fields(fields), "reverse": reverse, "cursor": _float(cursor), "limit": limit
        })

    def get_event_reason(self, entry_id, with_reason=False, with_error=False):
        """
        Returns a human-friendly reason for some event that has been caused by an action described by the specified
        audit log entry.
        """

        fields = ["issuer", "type", "status"]
        if with_reason:
            fields.append("reason")
        if with_error:
            fields.append("error")

        entry = self.get_audit_log_entry(entry_id, fields=fields)

        issuer = "Wall-E" if entry["issuer"] == "wall-e" else entry["issuer"]
        reason = "{type} by {issuer} ({status})".format(issuer=issuer, type=entry["type"], status=entry["status"])
        if with_error and "error" in entry:
            reason += ": " + entry["error"]
        elif with_reason and "reason" in entry:
            reason += ": " + entry["reason"]

        return reason

    def _get_audit_log(self, issuer=None, event_type=None, project=None, host_inv=None, host_name=None, host_uuid=None,
                       automation_plot=None, scenario_id=None, start_time=None, end_time=None,
                       fields=None, reverse=True, cursor=None, limit=None, status=None):
        return self.__api.get("/audit-log", params={
            "issuer": issuer, "type": event_type, "project": project,
            "host_inv": host_inv, "host_name": host_name, "host_uuid": host_uuid,
            "automation_plot": automation_plot, "scenario_id": scenario_id,
            "start_time": _float(start_time), "end_time": _float(end_time),
            "fields": _fields(fields), "reverse": reverse, "cursor": _float(cursor), "limit": limit, "status": status
        })

    def get_deploy_log(self, host_id, provisioner=None, tail_bytes=None):
        return self.__api.get(self.__host_uri(host_id, "/deploy-log"),
                              params={"provisioner": provisioner, "tail_bytes": tail_bytes}, stream=True).iter_lines()

    def get_profile_log(self, host_id):
        return self.__api.get(self.__host_uri(host_id, "/profile-log"), stream=True).iter_lines()

    def get_power_status(self, host_id):
        return self.__api.get(self.__host_uri(host_id, "/power-status"))

    def get_deploy_configs(self, provisioner=None):
        return self.__api.get("/deploy-configs", params={"provisioner": provisioner})

    def get_settings(self):
        return self.__api.get("/settings")

    def set_settings(self, disable_automation=None, disable_healing=None, disable_dns_automation=None,
                     enable_fsm_handbrake=None, fsm_handbrake_timeout=None,
                     enable_scenario_fsm_handbrake=None, scenario_fsm_handbrake_timeout=None,
                     inventory_invalid_hosts_limit=None, **settings):
        known_params = _drop_none({
            "disable_automation": disable_automation,
            "disable_healing": disable_healing, "disable_dns_automation": disable_dns_automation,
            "enable_fsm_handbrake": enable_fsm_handbrake, "fsm_handbrake_timeout": fsm_handbrake_timeout,
            "enable_scenario_fsm_handbrake": enable_scenario_fsm_handbrake, "scenario_fsm_handbrake_timeout": scenario_fsm_handbrake_timeout,
            "inventory_invalid_hosts_limit": inventory_invalid_hosts_limit
        })
        return self.__api.post("/settings", dict(known_params, **settings))

    def get_authorization_url(self):
        return self.__api.get("/access-token")["authorization_url"]

    def obtain_access_token(self, authorization_code):
        return self.__api.post("/access-token", request={"authorization_code": authorization_code})["access_token"]

    def get_scenarios(self, scenario_id=None, name=None, scenario_type=None,
                      offset=None, limit=None, ticket_key=None, issuer=None, fields=None, status=None):
        params = _drop_none({
            "scenario_id": scenario_id, "name": name, "scenario_type": scenario_type,
            "offset": offset, "limit": limit, "ticket_key": ticket_key,
            "issuer": issuer, "status": status, "fields": _fields(fields)
        })

        return self.__api.get("/scenarios", params=params)

    def create_add_hosts_rtc_scenario(self, name, ticket_key, hosts, target_project_id,
                                      responsible=None, ticket_created_by=None,
                                      labels=None, autostart=False):
        params = _drop_none({
            "name": name,  "ticket_key": ticket_key, "autostart": autostart,
            "hosts": hosts, "target_project_id": target_project_id,
            "labels": _drop_none(dict(labels or {},
                                     responsible=responsible,
                                     ticket_created_by=ticket_created_by)) or None,
        })
        return self.__api.post("/scenarios/hosts_add_rtc", request=params)

    def add_scenario(self, name=None, scenario_type=None, hosts=None, script_args=None,
                     ticket_key=None, labels=None, autostart=False, reason=None):
        params = _drop_none({
            "name": name, "scenario_type": scenario_type, "hosts": hosts,
            "script_args": script_args, "ticket_key": ticket_key, "labels": labels,
            "autostart": autostart, "reason": reason,
        })
        return self.__api.post("/scenarios", request=params)

    def iter_scenarios(self, fields=None, limit=None, offset=0, **kwargs):
        while limit is None or offset < limit:
            response = self.get_scenarios(offset=offset, limit=limit, fields=fields, **kwargs)
            scenarios = response["result"]
            if not scenarios:
                break
            for scenario in scenarios:
                yield scenario
            offset += len(scenarios)

    def get_scenario(self, scenario_id, fields=None):
        url = "/scenarios/{}".format(scenario_id)
        return self.__api.get(url, params={"fields": _fields(fields)})

    def start_scenario(self, scenario_id, reason):
        url = "/scenarios/{}/start".format(scenario_id)
        return self.__api.patch(url, request={"reason": reason}, params={})

    def modify_scenario(self, scenario_id, name=None, scenario_type=None, script_args=None,
                        ticket_key=None, labels=None, reason=None):
        url = "/scenarios/{}".format(scenario_id)
        request = _drop_none({
            "name": name, "scenario_type": scenario_type,
            "script_args": script_args, "ticket_key": ticket_key, "labels": labels,
            "reason": reason
        })
        return self.__api.patch(url, request=request)

    def cancel_scenario(self, scenario_id, reason):
        url = "/scenarios/{}/cancel".format(scenario_id)
        return self.__api.patch(url, request={"reason": reason}, params={})

    def pause_scenario(self, scenario_id, reason):
        url = "/scenarios/{}/pause".format(scenario_id)
        return self.__api.patch(url, request={"reason": reason}, params={})

    def get_scenarios_by_labels(self, scenario_id=None, name=None, scenario_type=None,
                                offset=None, limit=None, ticket_key=None, issuer=None, fields=None, labels=None):
        request = _drop_none({
            "labels": labels
        })
        params = _drop_none({
            "scenario_id": scenario_id, "name": name, "scenario_type": scenario_type,
            "offset": offset, "limit": limit, "ticket_key": ticket_key,
            "issuer": issuer, "fields": _fields(fields)
        })
        return self.__api.post("/scenarios/labels", request=request, params=params)

    def create_noc_maintenance(self, switch, name, ticket_key, reason=None):
        request = _drop_none({
            "switch": switch,
            "reason": reason,
            "ticket_key": ticket_key,
            "name": name
        })
        return self.__api.post("/maintenance/switch", request=request)

    def list_noc_maintenance(self, switch=None, scenario_id=None, name=None, issuer=None,
                             ticket_key=None, status=None, reverse=None, limit=None):
        params = _drop_none({
            "switch": switch,
            "scenario_id": scenario_id,
            "name": name,
            "issuer": issuer,
            "ticket_key": ticket_key,
            "status": status,
            "reverse": reverse,
            "limit": limit,
        })

        return self.__api.get("/maintenance/switch", params=params)

    def get_noc_maintenance_status(self, scenario_id):
        return self.__api.get("/maintenance/switch/{}".format(scenario_id))

    def start_noc_maintenance(self, scenario_id, reason=None):
        request = _drop_none({"reason": reason})
        return self.__api.post("/maintenance/switch/{}/start".format(scenario_id), request=request)

    def finish_noc_maintenance(self, scenario_id, reason=None):
        request = _drop_none({"reason": reason})
        return self.__api.post("/maintenance/switch/{}/finish".format(scenario_id), request=request)

    def cancel_noc_maintenance(self, scenario_id, reason=None):
        request = _drop_none({"reason": reason})
        return self.__api.post("/maintenance/switch/{}/cancel".format(scenario_id), request=request)

    def start_itdc_maintenance(self, scenario_id, reason=None):
        request = _drop_none({
            "reason": reason,
            "labels": {"WORK_COMPLETED": "true"},
        })
        return self.__api.post("/scenarios/{}".format(scenario_id), request=request)

    def iter_health_checks(self, limit=None, offset=0, **kwargs):
        while limit is None or offset < limit:
            response = self.get_health_checks(offset=offset, limit=limit, **kwargs)
            health_checks = response["result"]
            if not health_checks:
                break
            for health_check in health_checks:
                yield health_check
            offset += len(health_checks)

    def get_health_checks(self, fqdn=None, type=None, status=None, timestamp=None,
                          offset=None, limit=None):
        params = {
            "fqdn": fqdn, "type": type, "status": status, "timestamp": timestamp,
            "offset": offset, "limit": limit,
        }

        return self.__api.get("/health-checks", params=params)

    def get_project_role_members(self, project_id, role):
        return self.__api.get("/projects/{}/role/{}/members".format(project_id, role))

    def request_add_project_role_member(self, project_id, role, member=None):
        return self.__api.post("/projects/{}/role/{}/members".format(project_id, role),
                               request=_drop_none({"member": member}))

    def request_remove_project_role_member(self, project_id, role, member=None):
        return self.__api.delete("/projects/{}/role/{}/members".format(project_id, role),
                                 request=_drop_none({"member": member}))

    @classmethod
    def get_api_actions(cls):
        return list(cls.__api_action_http_methods.keys())

    @staticmethod
    def __host_uri(host_id, action=""):
        return "/hosts/{id}{action}".format(id=host_id, action=action)

    @staticmethod
    def __get_host_network_fields(fields=None):
        if fields is None:
            return None, None

        host_fields = set(fields) - constants.NETWORK_FIELDS
        network_fields = set(fields) & constants.NETWORK_FIELDS | {"uuid"}

        if network_fields:
            host_fields.add("uuid")

        return list(host_fields), list(network_fields)

    def __action(self, action, *args, **kwargs):
        return self.__api.call(self.__api_action_http_methods[action], *args, **kwargs)

    def get_checks_overrides(self):
        return self.__api.get("/settings/checks_percentage")

    def get_check_override(self, check):
        return self.__api.get("/settings/checks_percentage/{}".format(check))

    def set_check_override(self, check, percent):
        return self.__api.put("/settings/checks_percentage/{}".format(check), request={"percent": percent})

    def remove_check_override(self, check):
        return self.__api.delete("/settings/checks_percentage/{}".format(check))

    def get_global_timed_limits(self):
        return self.__api.get("/settings/global_timed_limits_overrides")

    def set_global_timed_limit(self, failure_name, period, limit):
        return self.__api.put(
            "/settings/global_timed_limits_overrides/{}".format(failure_name),
            request={"period": period, "limit": limit},
        )

    def remove_global_timed_limit(self, failure_name):
        return self.__api.delete("/settings/global_timed_limits_overrides/{}".format(failure_name))

    def get_shortnames(self):
        return self.__api.get("/shortnames")

    def get_one_shortname(self, path):
        return self.__api.get("/shortnames/{}".format(path))

    def add_shortname(self, path, name):
        return self.__api.post("/shortnames", request={"shortnames": [{"path": path, "name": name}]})

    def delete_shortname(self, path):
        return self.__api.delete("/shortnames/{}".format(path))

    # host network api
    def get_host_network(self, uuid=None, fields=None):
        return self.__api.get("/hosts/{0}/network".format(uuid), params={"fields": _fields(fields)})

    def get_hosts_network(self, uuids=None, fields=None, cursor=None, limit=None):
        params = {
            "fields": _fields(fields),
            "cursor": cursor,
            "limit": limit,
        }
        if uuids is None:
            raise Exception()
        return self.__api.post("/get-hosts/network", request={"uuids": uuids}, params=params)


class _Api(object):
    def __init__(self, url, name=None, access_token=None):
        self.__name = name
        self.__url = url
        self.__access_token = access_token

        adapter = requests.adapters.HTTPAdapter(
            max_retries=urllib3.util.retry.Retry(
                backoff_factor=0.5,
                allowed_methods=False,  # any method
                status_forcelist={429, 500, 502, 503, 504},
                total=3,  # any error type, 4 attempts total
            ),
        )
        self.__session = requests.Session()
        self.__session.mount(self.__url, adapter)

    def get(self, method, params=None, stream=False):
        return self.call("GET", method, params=params, stream=stream)

    def put(self, method, request, params=None):
        return self.call("PUT", method, params=params, request=request)

    def post(self, method, request, params=None):
        return self.call("POST", method, params=params, request=request)

    def patch(self, method, request, params=None):
        return self.call("PATCH", method, params=params, request=request)

    def delete(self, method, request=None, params=None):
        return self.call("DELETE", method, params=params, request=request)

    def call(self, http_method, api_method, params=None, request=None, stream=False):
        url = self.__url + "/v1" + api_method
        data = None

        params = params.copy() if params else {}
        params.setdefault("strict", True)

        user_agent = "Wall-E.Client/" + constants.version
        if self.__name:
            user_agent = self.__name + " " + user_agent

        headers = {"User-Agent": user_agent}

        if self.__access_token:
            headers["Authorization"] = "OAuth " + self.__access_token

        if request is not None:
            headers["Content-Type"] = "application/json"
            data = json.dumps(_drop_none(request))

        try:
            response = self.__session.request(http_method, url, params=params, headers=headers, data=data,
                                        stream=stream, timeout=60)
        except RequestException as e:
            raise WalleConnectionError("Request to {} failed: {}", url, e)

        if response.status_code not in (requests.codes.ok, requests.codes.created, requests.codes.no_content,
                                        requests.codes.accepted):
            error = _parse_response(response, error=True)

            if response.status_code == requests.codes.unauthorized:
                raise WalleAuthorizationError(error)
            else:
                print(response.status_code)
                print(error)
                raise WalleApiError(response.status_code, error)

        if stream:
            return response
        elif response.status_code == requests.codes.no_content:
            return None
        else:
            return _parse_response(response)


def _float(value):
    # Conversion of float to string looses precision
    #
    # >>> value = 1426596682.91458
    # >>> str(value), repr(value)
    # ('1426596682.91', '1426596682.91458')

    if value is not None:
        value = repr(value)

    return value


def _fields(fields):
    return None if fields is None else ",".join(fields)


def _drop_none(d):
    return {k: v for k, v in iteritems(d) if v is not None}


def _filter_dict_keys(d, keys):
    keys = set(keys)
    return {k: v for k, v in iteritems(d) if k in keys}


def _parse_response(response, error=False):
    try:
        reply = _get_json_response(response)
        if error:
            if not isinstance(reply, dict):
                raise ValueError

            message = reply.get("message")
            if not is_string(message) or not message:
                raise ValueError

            return message
        else:
            return reply
    except ValueError:
        if error:
            raise WalleConnectionError("Request to {} failed with an unknown error: {}.", response.url, response.reason)
        else:
            raise WalleConnectionError("Got an invalid reply for request to {}.", response.url)


def _get_json_response(response):
    content_type, type_options = cgi.parse_header(response.headers.get("Content-Type", ""))

    if content_type == "application/json":
        result = response.json()
    else:
        raise ValueError

    return result


PY3 = sys.version_info[0] == 3
if PY3:
    def is_string(value):
        return isinstance(value, str)

    def is_number(value):
        return isinstance(value, int)


    def iteritems(d, **kw):
        return iter(d.items(**kw))
else:
    def is_string(value):
        return isinstance(value, basestring)

    def is_number(value):
        return isinstance(value, (int, long))

    def iteritems(d, **kw):
        return d.iteritems(**kw)
