"""Various Wall-E constants."""

from __future__ import unicode_literals


version = "7.3.48"
"""Package version.

The variable name must be lowercase because of releaselib limitations.

Major number: for releases with broken API compatibility
Minor number: for releases that add new major features
Sub-minor number: for releases that fix or add new features
"""


class AutomationType:
    TYPE_ALL_AUTOMATION = "all"
    TYPE_HEALING_AUTOMATION = "healing"
    TYPE_DNS_AUTOMATION = "dns"

    ALL_AUTOMATION_TYPES = [TYPE_ALL_AUTOMATION, TYPE_HEALING_AUTOMATION, TYPE_DNS_AUTOMATION]

    AUTOMATION_TYPE_PATH_MAP = {
        TYPE_ALL_AUTOMATION: "",
        TYPE_HEALING_AUTOMATION: "/healing",
        TYPE_DNS_AUTOMATION: "/dns",
    }

    HEALING_AUTOMATION_LIMITS = ["max_unreachable_failures", "max_ssh_failures",
                                 "max_memory_failures", "max_disk_failures", "max_link_failures",
                                 "max_cpu_failures", "max_reboots_failures", "max_tainted_kernel_failures",
                                 "max_fs_check_failures", "max_dead_hosts"]
    DNS_AUTOMATION_LIMITS = ["max_dns_fixes"]

    AUTOMATION_TYPE_LIMIT_MAP = {
        TYPE_HEALING_AUTOMATION: HEALING_AUTOMATION_LIMITS,
        TYPE_DNS_AUTOMATION: DNS_AUTOMATION_LIMITS,
        TYPE_ALL_AUTOMATION: HEALING_AUTOMATION_LIMITS + DNS_AUTOMATION_LIMITS,
    }


class Provisioner:
    LUI = "lui"
    """Linux Unattended Installation (http://setup.search.yandex.net/)."""

    EINE = "eine"
    """Einstellung (http://eine.yandex-team.ru/)."""

    ALL = [LUI, EINE]
    """A list of available provisioners."""


class DeployNetwork:
    SERVICE = "service"
    """Deploy hosts in service (eine) network (VLAN 542)"""

    PROJECT = "project"
    """Deploy hosts in project's native vlan."""

    ALL = [SERVICE, PROJECT]


class MtnIpMethod:
    MAC = "mac"
    """
    Use mac-address for host-id part of mtn IP-address (like they do in RTC).
    IP-address is generated based upon the "mac-address/rtc" host-id scheme (net/64 + project-id + mac-address[:32]).
    """

    HOSTNAME = "hostname"
    """
    Use hostname for host-id part of mtn IP-address.
    IP-address is generated based upon the "hostname" host-id scheme (net/64 + project-id + md5(hostname)[:32]).
    """

    ALL = [MAC, HOSTNAME]


class HostState:
    FREE = "free"
    """Host is free and ready for assigning."""

    ASSIGNED = "assigned"
    """Host is assigned to a role."""

    MAINTENANCE = "maintenance"
    """Host is in maintenance mode."""

    PROBATION = "probation"
    """Host is being prepared for assignment."""

    ALL = [FREE, PROBATION, ASSIGNED, MAINTENANCE]
    """A list of states host can be in."""


class HostOperationState:
    OPERATION = "operation"
    """Host is in operation."""

    DECOMMISSIONED = "decommissioned"
    """Host had been decommissioned."""

    ALL = [OPERATION, DECOMMISSIONED]
    """A list of states host operation status can be in."""


class HostStatus:
    READY = "ready"
    """The host is ready."""

    MANUAL = "manual"
    """The host is marked for manual operation by administrator."""

    SWITCHING_PROJECT = "switching-project"
    """Host is moving from one project to another."""

    POWERING_ON = "powering-on"
    """Host is powering on now."""

    POWERING_OFF = "powering-off"
    """Host is powering off now."""

    REBOOTING = "rebooting"
    """Host is rebooting now."""

    CHECKING_DNS = "checking-dns"
    """Checking and fixing DNS records."""

    PROFILING = "profiling"
    """Host is profiling now."""

    PREPARING = "preparing"
    """Host is preparing now."""

    DEPLOYING = "deploying"
    """Host is deploying now."""

    DEACTIVATING = "deactivating"
    """Host is deactivating now."""

    DEAD = "dead"
    """Host is dead."""

    INVALID = "invalid"
    """Host has an inconsistent state which must be handled manually."""

    ADDING = "adding"
    """Host is adding now."""

    DELETING = "deleting"
    """Host is deleting now."""

    REPAIRING = "repairing"
    """Host is repairing now."""

    REPORTING_FAILURE = "reporting-failure"
    """Host is reporting failure now."""

    REPAIRING_RACK = "repairing-pack"
    """Host is repairing pack now."""

    REPAIRING_LINK = "repairing-link"
    """Host is repairing link now."""

    REPAIRING_BMC = "repairing-bmc"
    """Host is repairing bmc now."""

    RESETTING_BMC = "resetting-bmc"
    """Host is resetting bmc now."""

    CHANGING_MEMORY = "changing-memory"
    """Host is changing memory now."""

    CHANGING_DISK = "changing-disk"
    """Host is changing disk now."""

    REPLACING_DISK_CABLE = "replacing-disk-cable"
    """Host is replacing disk cable now."""

    REPAIRING_GPU = "repairing-gpu"
    """Host is repairing gpu now."""

    REPAIRING_OVERHEAT = "repairing-overheat"
    """Host is repairing overheat now."""

    REPAIRING_CAPPING = "repairing-capping"
    """Host is repairing capping now."""

    REPAIRING_REBOOTS = "repairing-reboots"
    """Host is repairing reboots now."""

    SWITCHING_TO_MAINTENANCE = "switching-to-maintenance"
    """Host is switching to maintenance now."""

    SWITCHING_TO_ASSIGNED = "switching-to-assigned"
    """Host is switching to assigned now."""

    SWITCHING_VLANS = "switching-vlans"
    """Host is switching vlans now."""

    SYNCING_BOT_PROJECT = "syncing-bot-project"
    """Host is syncing bot project now."""

    ALL = [READY, MANUAL, SWITCHING_PROJECT, POWERING_ON, POWERING_OFF, REBOOTING, CHECKING_DNS,
           PROFILING, PREPARING, DEPLOYING, DEACTIVATING, DEAD, INVALID, ADDING, DELETING, REPAIRING, REPORTING_FAILURE,
           REPAIRING_RACK, REPAIRING_LINK, REPAIRING_BMC, RESETTING_BMC, CHANGING_MEMORY, CHANGING_DISK,
           REPLACING_DISK_CABLE, REPAIRING_GPU, REPAIRING_OVERHEAT, REPAIRING_CAPPING, REPAIRING_REBOOTS,
           SWITCHING_TO_MAINTENANCE, SWITCHING_TO_ASSIGNED, SWITCHING_VLANS, SYNCING_BOT_PROJECT]
    """A list of statuses host can be in."""


class Restriction:
    EXPERT = "expert"
    """Allows to work with hosts that aren't configured to work with Expert System.
    All health checks are skipped for hosts with this restriction."""

    REBOOT = "reboot"
    """Prohibit reboot."""

    PROFILE = "profile"
    """Prohibit profiling."""

    REDEPLOY = "redeploy"
    """Prohibit redeployment."""

    AUTOMATED_DNS = "automated-dns"
    """Prohibit automated DNS modification."""

    AUTOMATED_REBOOT = "automated-reboot"
    """Prohibit automated reboot."""

    AUTOMATED_PROFILE = "automated-profile"
    """Prohibit automated profiling."""

    AUTOMATED_REDEPLOY = "automated-redeploy"
    """Prohibit automated redeployment."""

    AUTOMATED_MEMORY_CHANGE = "automated-memory-change"
    """Prohibit automated change of a corrupted memory."""

    AUTOMATED_DISK_CHANGE = "automated-disk-change"
    """Prohibit automated change of a corrupted disk."""

    AUTOMATED_DISK_CABLE_REPAIR = "automated-disk-cable-repair"
    """Prohibit automated repair of malfunctioning disk cable."""

    AUTOMATED_LINK_REPAIR = "automated-link-repair"
    """Prohibit automated repair of malfunctioning link."""

    AUTOMATED_CPU_REPAIR = "automated-cpu-repair"
    """Prohibit automated CPU repairing."""

    AUTOMATED_REBOOTS_REPAIR = "automated-reboots-repair"
    """Prohibit automated repairing of unexpected reboots."""

    AUTOMATED_REPAIRING = "automated-repairing"
    """Prohibit all automated hardware repairing."""

    AUTOMATED_HEALING = "automated-healing"
    """Prohibit automated host healing."""

    AUTOMATION = "automation"
    """Prohibit any automation."""

    ALL = [REBOOT, PROFILE, REDEPLOY, AUTOMATED_DNS, AUTOMATED_HEALING, AUTOMATION, EXPERT,
           AUTOMATED_MEMORY_CHANGE, AUTOMATED_DISK_CHANGE, AUTOMATED_LINK_REPAIR, AUTOMATED_DISK_CABLE_REPAIR,
           AUTOMATED_CPU_REPAIR, AUTOMATED_REBOOTS_REPAIR, AUTOMATED_REPAIRING, AUTOMATED_REBOOT, AUTOMATED_PROFILE,
           AUTOMATED_REDEPLOY]
    """A list of available restrictions."""


class CmsApiVersion:
    """Supported versions of CMS-API protocol.
    If your version of wall-e.api have an outdated list,
    you may just pass version string instead or update wall-e.api."""
    V1_0 = "v1.0"
    V1_1 = "v1.1"
    V1_2 = "v1.2"
    V1_3 = "v1.3"
    V1_4 = "v1.4"
    ALL = [V1_0, V1_1, V1_2, V1_3, V1_4]


class CmsTaskAction:
    REBOOT = "reboot"
    PROFILE = "profile"
    TEMPORARY_UNREACHABLE = "temporary-unreachable"

    ALL = [REBOOT, PROFILE, TEMPORARY_UNREACHABLE]


class VlanScheme:
    """Supported VLAN-schemes.
    If your version of wall-e.api have an outdated list,
    you may just pass a plain string instead or update wall-e.api."""
    STATIC = "static"
    SEARCH = "search"
    MTN = "mtn"
    ALL = [STATIC, MTN, SEARCH]


class SshOptions:
    FORBID = "forbid"
    """Forbid using of ssh for operation."""

    FALLBACK = "fallback"
    """Allow using of ssh for operation with fallback to IPMI."""

    ONLY = "only"
    """Perform operation via ssh."""

    ALL = [FORBID, FALLBACK, ONLY]
    """List of all available ssh options."""


class CheckType:
    UNREACHABLE = "UNREACHABLE"
    SSH = "ssh"
    META = "META"
    W_META = "walle_meta"
    MEMORY = "walle_memory"
    DISK = "walle_disk"
    LINK = "walle_link"
    BMC = "walle_bmc"
    GPU = "walle_gpu"
    INFINIBAND = "walle_infiniband"
    REBOOTS = "walle_reboots"
    TAINTED_KERNEL = "walle_tainted_kernel"
    CPU = "walle_cpu"
    CPU_CACHES = "walle_cpu_caches"
    CPU_THERMAL = "walle_cpu_thermal"
    CPU_CAPPING = "walle_cpu_capping"
    FS_CHECK = "walle_fs_check"
    NETMON = "netmon"
    WALLE_RACK = "walle_rack"

    ALL = [UNREACHABLE, SSH, META, W_META, MEMORY, DISK, LINK, BMC, GPU, REBOOTS, TAINTED_KERNEL,
           CPU, CPU_CACHES, CPU_THERMAL, CPU_CAPPING, FS_CHECK, NETMON, WALLE_RACK, INFINIBAND]


class CheckStatus:
    PASSED = "passed"
    """Check passed."""

    SUSPECTED = "suspected"
    """Host marked suspected because check failed for host but it's too early to mark it failed."""

    FAILED = "failed"
    """Host marked failed because check failed."""

    MISSING = "missing"
    """Check data is missing - Wall-E has received a NO DATA check."""

    VOID = "void"
    """Check is missing, Wall-E has not received check data."""

    INVALID = "invalid"
    """Check has invalid metadata."""

    STALED = "staled"
    """Check is staled."""

    UNSURE = "unsure"
    """Check data is not fresh enough, need to wait for check status update."""

    ALL = [PASSED, SUSPECTED, FAILED, MISSING, VOID, INVALID, STALED, UNSURE]


NETWORK_FIELDS = {
    "active_mac_time",
    "network_timestamp",
    "ips_time"
}


PRODUCTION_URL = "https://api.wall-e.yandex-team.ru"
"""Wall-E production host."""

TESTING_URL = "https://api.wall-e-test.yandex-team.ru"
"""Wall-E testing host."""
