package main

import (
	"bytes"
	"encoding/json"
	"fmt"
	"io"
	"log"
	"net/http"
	"os"
	"os/exec"
)

const (
	project                    = "yc.wall-e.cloud"
	cluster                    = "yc.wall-e.main-folder"
	missingCheckAlertID        = "missing-%s-checks"
	missingChecksCmd           = "missing-checks"
	activeChecksAgeID          = "active-checks-age"
	activeChecksAgeCmd         = "active-checks-age"
	expertiseAgeID             = "expertise-age"
	expertiseAgeCmd            = "expertise-age"
	invalidHealthChecksAlertID = "invalid-%s-health-checks"
	invalidHealthChecksCmd     = "invalid-health-checks"
)

const alertTemplate = `
{
  "id": "%s",
  "name": "%s",
  "version": %d,
  "channels": [
    {
      "id": "e9188128-2231-4648-8b6b-4ba9fdc798b5",
      "config": {
        "notifyAboutStatuses": [
          "WARN",
          "ALARM",
          "OK"
        ],
        "repeatDelaySecs": 0
      }
    }
  ],
  "type": {
    "expression": {
      "program": "%s",
      "checkExpression": ""
    }
  },
  "annotations": {},
  "windowSecs": 300,
  "delaySecs": 0,
  "description": "",
  "resolvedEmptyPolicy": "RESOLVED_EMPTY_WARN",
  "noPointsPolicy": "NO_POINTS_DEFAULT",
  "labels": {},
  "serviceProviderAnnotations": {},
  "severity": "SEVERITY_UNSPECIFIED"
}
`

const programTemplateMissingChecks = "let all_missing=series_sum({project='" + project + "',cluster='" + cluster +
	"',service='walle_hosts',sensor=~'hosts.total.%s-checks-(missing|void|staled)'});" +
	"let value=last(all_missing);" +
	"warn_if(value>%d);" +
	"alarm_if(value>%d);"

const programTemplateActiveChecksAge = "let percentile=histogram_percentile(99, " +
	"{project='" + project + "',cluster='" + cluster +
	"',service='walle_health',sensor=~'health.health_timestamp_age_(UNREACHABLE|ssh)'});" +
	"let value=last(percentile/60);" +
	"warn_if(value>%d);" +
	"alarm_if(value>%d);"

const programTemplateExpertiseAge = "let lines={project='" + project + "',cluster='" + cluster +
	"',service='walle_hosts',sensor='hosts.expertize-age-99'};" +
	"let value=last(lines)/60;" +
	"warn_if(value>%d);" +
	"alarm_if(value>%d);"

const programTemplateInvalidHealthChecks = "let lines={project='" + project + "',cluster='" + cluster +
	"',service='walle_hosts',sensor='hosts.total-health-check.%s-invalid'};" +
	"let value=last(lines);" +
	"warn_if(value>%d);" +
	"alarm_if(value>%d);"

var apiURL = fmt.Sprintf("https://solomon.cloud.yandex-team.ru/api/v2/projects/%s/alerts/", project)

var usage = `
Usage:
	generate-alerts <command>

Commands:
%s
`

func init() {
	var cmds = ""
	for _, cmd := range []string{missingChecksCmd, activeChecksAgeCmd, expertiseAgeCmd, invalidHealthChecksCmd} {
		cmds += fmt.Sprintf("\t%s\n", cmd)
	}
	usage = fmt.Sprintf(usage, cmds)
}

type solomonResponse struct {
	Version int `json:"version"`
}

func main() {
	if len(os.Args) != 2 {
		fmt.Println(usage)
		os.Exit(1)
	}
	thresholds := getThresholds()
	iamTokenRaw, err := exec.Command("yc", "iam", "create-token").Output()
	if err != nil {
		log.Fatal(err)
	}
	iamToken := string(iamTokenRaw[:len(iamTokenRaw)-1])
	client := newClient(iamToken)
	switch os.Args[1] {
	case missingChecksCmd:
		for _, checkGroup := range []string{"walle", "active", "passive", "netmon"} {
			client.tryUpdate(
				fmt.Sprintf(missingCheckAlertID, checkGroup),
				fmt.Sprintf(
					programTemplateMissingChecks,
					checkGroup,
					thresholds.Missing[checkGroup].Warn,
					thresholds.Missing[checkGroup].Alarm,
				),
			)
			fmt.Println()
		}
	case activeChecksAgeCmd:
		client.tryUpdate(activeChecksAgeID, fmt.Sprintf(
			programTemplateActiveChecksAge,
			thresholds.Age["active-checks"].Warn,
			thresholds.Age["active-checks"].Alarm,
		))
	case expertiseAgeCmd:
		client.tryUpdate(expertiseAgeID, fmt.Sprintf(
			programTemplateExpertiseAge,
			thresholds.Age["expertise"].Warn,
			thresholds.Age["expetise"].Alarm,
		))
	case invalidHealthChecksCmd:
		for _, healthCheck := range getHealthChecks() {
			client.tryUpdate(
				fmt.Sprintf(invalidHealthChecksAlertID, healthCheck),
				fmt.Sprintf(
					programTemplateInvalidHealthChecks,
					healthCheck,
					thresholds.Missing["any"].Warn,
					thresholds.Missing["any"].Alarm,
				),
			)
			fmt.Println()
		}
	default:
		fmt.Println(usage)
		os.Exit(1)
	}
}

type solomonClient struct {
	client   *http.Client
	iamToken string
}

func newClient(iamToken string) *solomonClient {
	return &solomonClient{client: &http.Client{}, iamToken: iamToken}
}

func (c *solomonClient) newRequest(method, url string, body io.Reader) *http.Request {
	req, err := http.NewRequest(method, url, body)
	if err != nil {
		log.Fatal(err)
	}
	req.Header.Add("Content-Type", "application/json")
	req.Header.Add("Accept", "application/json")
	req.Header.Add("Authorization", "Bearer "+c.iamToken)
	return req
}

func (c *solomonClient) tryUpdate(id, program string) {
	version := c.getAlertVersion(id)
	alert := fmt.Sprintf(alertTemplate, id, id, version, program)
	updateStatus := c.updateAlert(id, alert)
	if updateStatus == http.StatusNotFound {
		c.createAlert(id, alert)
	}
}

func (c *solomonClient) getAlertVersion(id string) int {
	request := c.newRequest("GET", apiURL+id, nil)
	resp, err := c.client.Do(request)
	if err != nil {
		log.Fatal(err)
	}
	alertInfo := solomonResponse{}
	dec := json.NewDecoder(resp.Body)
	if err = dec.Decode(&alertInfo); err != nil {
		log.Fatal(err)
	}
	defer resp.Body.Close()
	return alertInfo.Version
}

func (c *solomonClient) updateAlert(id, alert string) int {
	resp, status := c.pushAlertData(apiURL+id, "PUT", alert)
	fmt.Println("update alert", id, ":\n", resp)
	return status
}

func (c *solomonClient) createAlert(id, alert string) {
	_, resp := c.pushAlertData(apiURL, "POST", alert)
	fmt.Println("create new alert", id, ":\n", resp)

}

func (c *solomonClient) pushAlertData(url, method, alert string) (string, int) {
	request := c.newRequest(method, url, bytes.NewBuffer([]byte(alert)))
	resp, err := c.client.Do(request)
	if err != nil {
		log.Fatal(err)
	}
	respBody, err := io.ReadAll(resp.Body)
	if err != nil {
		log.Fatal(err)
	}
	defer resp.Body.Close()
	return string(respBody), resp.StatusCode
}

func getThresholds() thresholdMaps {
	f, err := os.Open("thresholds.json")
	if err != nil {
		log.Fatal(err)
	}
	defer f.Close()
	dec := json.NewDecoder(f)
	res := thresholdMaps{}
	if err = dec.Decode(&res); err != nil {
		log.Fatal(err)
	}
	return res
}

type thresholdMaps struct {
	Missing map[string]threshold `json:"missing-checks"`
	Age     map[string]threshold `json:"age"`
	Invalid map[string]threshold `json:"invalid-health-checks"`
}

type threshold struct {
	Warn  int `json:"warn"`
	Alarm int `json:"alarm"`
}

func getHealthChecks() []string {
	resp, err := http.Get("https://api.wall-e.yandex-team.ru/v1/constants")
	if err != nil {
		log.Fatal(err)
	}
	constants := struct {
		JugglerChecks []string `json:"juggler_checks"`
	}{}
	dec := json.NewDecoder(resp.Body)
	if err = dec.Decode(&constants); err != nil {
		log.Fatal(err)
	}
	defer resp.Body.Close()
	return constants.JugglerChecks
}
