import re
import os
import time
from contextlib import contextmanager


BACKUP_NAME_REGEX = re.compile(
    r"^backup_(.+)_([^_]+)_([0-9]+).arz$"
)


class BackupInfo:

    def __init__(self, storage, name, host, ts):
        self.storage = storage
        self.name = name
        self.host = host
        self.ts = ts

    def get(self):
        return self.storage.get_backup_file(self.name)

    def __str__(self):
        return self.name

    def __gt__(self, other):
        if not isinstance(other, BackupInfo):
            raise TypeError
        return self.ts > other.ts

    def __ge__(self, other):
        if not isinstance(other, BackupInfo):
            raise TypeError
        return self.ts >= other.ts

    def __lt__(self, other):
        if not isinstance(other, BackupInfo):
            raise TypeError
        return self.ts < other.ts

    def __le__(self, other):
        if not isinstance(other, BackupInfo):
            raise TypeError
        return self.ts <= other.ts

    def __eq__(self, other):
        if not isinstance(other, BackupInfo):
            raise TypeError
        return self.ts == other.ts


class MongoBackupBaseStorage:

    def __init__(self, hostname, rs_name, root_dir):
        self.hostname = hostname
        self.rs_name = rs_name
        self.root_dir = root_dir

    @contextmanager
    def save_backup(self, backup_info=None):
        if not backup_info:
            backup_info = self.get_new_backup_info()
        filepath = os.path.join(self.root_dir, backup_info.name)
        try:
            yield backup_info
        finally:
            self._store(backup_info.name, filepath)

    def get_new_backup_info(self):
        ts = int(time.time())
        name = "backup_{}_{}_{}.arz".format(
            self.rs_name,
            self.hostname,
            ts
        )
        return BackupInfo(
            self, name,
            self.hostname,
            ts
        )

    def find_last_backup(self, host=None):
        for backup in self.list():
            if not host or backup.host == host:
                return backup
        return None

    def _list(self):
        raise NotImplementedError()

    def _del(self, name):
        raise NotImplementedError()

    def _store(self, filename, filepath):
        raise NotImplementedError()

    def _get(self, name):
        raise NotImplementedError()

    def get_backup_file(self, name):
        return self._get(name)

    def list(self):
        result = []
        for filename in self._list():
            match = BACKUP_NAME_REGEX.match(filename)
            if not match:
                continue
            rs_name = match.group(1)
            host = match.group(2)
            ts = int(match.group(3))
            if rs_name != self.rs_name:
                continue
            result.append(BackupInfo(
                self,
                filename,
                host,
                ts
            ))
        result.sort(reverse=True)
        return result

    def clean(self, keep=15):
        for backup in self.list():
            if keep >= 0:
                keep -= 1
                continue
            self._del(backup.name)
