import subprocess
import pymongo


PRIMARY = 1
SECONDARY = 2


def execute_command(*args):
    try:
        return subprocess.check_output([str(i) for i in args]).decode('utf-8')
    except subprocess.CalledProcessError as e:
        raise AssertionError('{} returned with error {}'.format(args[0], e.output))


def mongo_dump(target_path, mongodump_path="mongodump", host="localhost", port=27017,
               username=None, password=None, authentication_database="admin"):

    command = [mongodump_path, "--host", "\"{0}\"".format(host), "--port", port, "--oplog", "--gzip",
               "--archive=\"{0}\"".format(target_path)]

    if username is not None:
        command.extend(["--username", username])
    if password is not None:
        command.extend(["--password", password])
    if authentication_database is not None:
        command.extend(["--authenticationDatabase", authentication_database])

    return execute_command(*command)


def restore(source_path, mongorestore_path="mongorestore", host="localhost", port=27017,
            include=None, exclude=None,
            username=None, password=None, authentication_database=None):

    command = [mongorestore_path, "--host", host, "--port", port, "--gzip",
               f"--archive=\"{source_path}\""]
    if username is not None:
        command.extend(["--username", username])
    if password is not None:
        command.extend(["--password", password])
    if authentication_database is not None:
        command.extend(["--authenticationDatabase", authentication_database])

    if exclude:
        command.append("--nsExclude={}".format(exclude))

    if include:
        command.append("--nsInclude={}".format(include))

    if not include and not exclude:
        command.append("--oplogReplay")

    return execute_command(*command)


def _make_local_connection_uri(host="localhost", port=27107,  username=None, password=None):
    host = "{}:{}".format(host, port)
    return "mongodb://{}:{}@{}/admin".format(
        username,
        password,
        host
    )


def get_mongodb_status(host="localhost", port=27107,  username=None, password=None):
    uri = _make_local_connection_uri(
        host, port, username, password
    )
    client = pymongo.MongoClient(uri)
    return client.admin.command("replSetGetStatus")


def is_primary(status):
    for member in status["members"]:
        if member.get("self", False) and member["state"] == PRIMARY:
            return True
    return False


def is_secondary(status):
    for member in status["members"]:
        if member.get("self", False) and member["state"] == SECONDARY:
            return True
    return False


def get_state(status):
    for member in status["members"]:
        if member.get("self", False):
            return member["stateStr"]
    return None


def get_oplag(status):
    master_date = None
    self_date = None
    for member in status["members"]:
        if member.get("self", False):
            self_date = member["optimeDate"]
        if member["state"] == PRIMARY:
            master_date = member["optimeDate"]
    return (master_date - self_date).total_seconds()
