from infra.walle.database.mongo_config import MongoConfig
from infra.walle.database.backup_storage.backup_storage_s3 import MongoBackupS3Storage
from infra.walle.database.backup_storage.backup_storage_local import MongoBackupLocalStorage
from infra.walle.database.clients import mongo_tools
from infra.walle.database.clients.juggler import notify_backup_completed
import logging
import time

log = logging.getLogger(__name__)


class ReplicaIsMaster(Exception):
    pass


class MongoBackupDaemon:

    def __init__(self, config: MongoConfig):
        self.config = config
        self.s3_storage = MongoBackupS3Storage(
            self.config.self_host,
            self.config.mongo_rs_name,
            tmp_dir=self.config.local_storage,
            s3_url=self.config.s3_url,
            bucket=self.config.s3_bucket,
            access_key=self.config.s3_access_key,
            secret_key=self.config.s3_secret_key
        )

        self.local_storage = MongoBackupLocalStorage(
            self.config.self_host,
            self.config.mongo_rs_name,
            root_dir=self.config.local_storage
        )

    def run(self):
        while True:
            self._tick()
            time.sleep(self.config.backup_period)

    def run_once(self):
        self._tick()

    def clean_tmp(self):
        pass

    def make_backup(self):
        with self.local_storage.save_backup() as backup_info:
            mongo_tools.mongo_dump(
                backup_info.get(), self.config.mongodump_path, self.config.self_host, self.config.self_port,
                self.config.mongo_username, self.config.mongo_password, self.config.mongo_admin_db
            )
            return backup_info

    def store_backup(self, backup_info):
        with self.s3_storage.save_backup(backup_info) as s3_backup:
            return s3_backup

    def check_secondary(self):
        status = mongo_tools.get_mongodb_status(
            self.config.self_host,
            self.config.self_port,
            self.config.mongo_username,
            self.config.mongo_password
        )
        if mongo_tools.is_primary(status):
            raise ReplicaIsMaster()

    def cleanup(self):
        self.s3_storage.clean()
        self.local_storage.clean()

    def notify_juggler(self):
        notify_backup_completed(
            self.config.self_host,
            self.config.self_port,
            self.config.mongo_rs_name
        )

    def execute_backup_routine(self):
        log.info("Checking if replica is secondary...")
        self.check_secondary()
        log.info("Creating backup in local storage...")
        local_backup = self.make_backup()
        log.info("Made backup: {}".format(local_backup))
        log.info("Storing in s3...")
        s3_backup = self.store_backup(local_backup)
        log.info("Backup stored in s3: {}".format(s3_backup))
        log.info("Cleaning up...")
        self.cleanup()
        log.info("Notifying juggler...")
        self.notify_juggler()
        log.info("All done.")

    def _tick(self):
        try:
            self.execute_backup_routine()
        except ReplicaIsMaster:
            log.info("replica is master, skipping")
        except Exception as e:
            log.exception("Exception while creating backup: %s", e)
