import argparse
from infra.walle.database.mongo_backup_restore.mongo_restore.restore import MongoBackupRestorer
from infra.walle.database.mongo_config import MongoConfig
from infra.walle.database.util import add_subcommand, pretty_print
import logging

logging.basicConfig(level=logging.INFO)
log = logging.getLogger(__name__)


def _fetch(args):
    restorer = get_restorer(args)
    found_backup = None
    for backup in restorer.list_l3():
        if backup.name == args.backup:
            log.info("Found s3 backup: {}".format(backup.name))
            found_backup = backup
            break
    if found_backup:
        log.info("Fetching backup: {}".format(found_backup.name))
        path = found_backup.get()
        log.info("Fetched to: {}".format(path))
    else:
        log.error("Backup {} not found".format(args.backup))


def _restore_backup(args):
    restorer = get_restorer(args)
    source_backup_local = None
    source_backup_s3 = None
    if args.local or not args.s3:
        for backup in restorer.list_local():
            if backup.name == args.backup:
                source_backup_local = backup
                break
    if args.s3 or not args.local:
        for backup in restorer.list_l3():
            if backup.name == args.backup:
                source_backup_s3 = backup
                break
    if source_backup_local:
        log.info("Using local backup: {}".format(source_backup_local.name))
        restorer.restore(source_backup_local, args.ns_include, args.ns_exclude)
    elif source_backup_s3:
        log.info("Using s3 backup: {}".format(source_backup_local.name))
        restorer.restore(source_backup_s3, args.ns_include, args.ns_exclude)
    else:
        log.error("Backup: {} not found".format(args.backup))


def _list_backups(args):
    restorer = get_restorer(args)
    if args.local or not args.s3:
        pretty_print("Local backups:")
        for backup in restorer.list_local():
            pretty_print(backup)
    if args.s3 or not args.local:
        pretty_print("S3 backups:")
        for backup in restorer.list_l3():
            pretty_print(backup)


def get_restorer(args):
    return MongoBackupRestorer(MongoConfig(args))


def main():
    root_parser = argparse.ArgumentParser(description='Wall-e mongo backup restore tool.')
    root_parser.set_defaults(func=root_parser.print_help)
    sub = root_parser.add_subparsers(required=True, dest="subcommand")

    restore_subcommand = add_subcommand(sub, 'restore', _restore_backup, 'Restores specific backup')
    restore_subcommand.add_argument("backup", help="Backup name to restore from")
    restore_subcommand.add_argument('-t', '--host', type=str,
                                    nargs='?',
                                    dest="host",
                                    default=None,
                                    help='Target host to restore to, default: self')
    restore_subcommand.add_argument('-p', '--port', type=int,
                                    nargs='?',
                                    dest="port",
                                    default=None,
                                    help='Target port to restore to, default: self')
    restore_subcommand.add_argument('-l', '--local', dest="local", help='Use only local storage')
    restore_subcommand.add_argument('-3', '--s3', dest="s3", help='use only s3 storage')
    restore_subcommand.add_argument('-i', '--ns_include', type=str,
                                    nargs='?',
                                    dest="ns_include",
                                    default=None,
                                    help='Include namespaces')
    restore_subcommand.add_argument('-e', '--ns_exclude', type=str,
                                    nargs='?',
                                    dest="ns_exclude",
                                    default=None,
                                    help='Exclude namespaces')

    fetch_subcommand = add_subcommand(sub, 'fetch', _fetch, 'Fetches specific backup from s3')
    fetch_subcommand.add_argument("backup", help="Backup name to fetch")

    list_subcommand = add_subcommand(sub, 'list', _list_backups, 'Lists available backups')
    list_subcommand.add_argument('-l', '--local',  action='store_true',
                                 dest="local", help='List backups from local storage')
    list_subcommand.add_argument('-3', '--s3',  action='store_true', dest="s3", help='List backups from S3')

    args = root_parser.parse_args()
    args.func(args)
    pass


if __name__ == "__main__":
    main()
