from infra.walle.database.mongo_config import MongoConfig
from infra.walle.database.clients.mongo_tools import get_mongodb_status, is_primary, is_secondary, get_oplag, get_state
from infra.walle.database.clients.juggler import notify_event, CheckStatus


class MongoCheck:

    def __init__(self, name, action, main=False):
        self.name = name
        self.action = action
        self.main = main


class MongoMonitor:

    def get_status(self):
        return get_mongodb_status(self.config.self_host, self.config.self_port,
                                  self.config.mongo_username, self.config.mongo_password)

    def _check_master(self):
        if is_primary(self.get_status()):
            return CheckStatus.OK, f"{self.config.self_host}:{self.config.self_port} is primary replica of {self.config.mongo_rs_name}"
        return None, None

    def _check_oplag(self):
        oplag = get_oplag(self.get_status())
        if oplag > self.config.max_oplag / 2:
            desription = f"{self.config.self_host}:{self.config.self_port}'s oplag of rs {self.config.mongo_rs_name} is too high: {oplag} seconds."
            if oplag > self.config.max_oplag:
                return CheckStatus.CRIT, desription
            else:
                return CheckStatus.WARN, desription
        else:
            return CheckStatus.OK, f"{self.config.self_host}:{self.config.self_port}'s oplag of rs {self.config.mongo_rs_name} is OK: {oplag} seconds."

    def _check_status(self):
        status = self.get_status()
        state_str = get_state(status)
        description = f"{self.config.self_host}:{self.config.self_port} is {state_str} replica of {self.config.mongo_rs_name}"
        if is_primary(status) or is_secondary(status):
            return CheckStatus.OK, description
        else:
            return CheckStatus.CRIT, description

    def return_statuses(self):
        for check in self.checks:
            status, description = check.action()
            yield check.name, status, description

    def monitor(self):
        main_status = True
        for check in self.checks:
            status, description = check.action()
            if check.main and status == CheckStatus.CRIT:
                main_status = False
            notify_event(self.config.self_host, self.config.self_port, self.config.mongo_rs_name,
                         check.name, status, description)
        return main_status

    def __init__(self, config: MongoConfig):
        self.config = config
        self.checks = [
            MongoCheck("master", self._check_master),
            MongoCheck("oplag", self._check_oplag),
            MongoCheck("status", self._check_status, main=True)
        ]
