# coding: utf-8
"""
Storable items for authentication and authorization mechanisms.
"""
from enum import Enum
from mongoengine.document import EmbeddedDocument, Document
from mongoengine.fields import StringField, ListField, EmbeddedDocumentField

from sepelib.mongo.fields import EnumField


class StaffGroupsCache(Document):
    """Persistent cache for user staff groups."""
    login = StringField(primary_key=True)
    group_ids = ListField(StringField())

    @classmethod
    def set(cls, login, group_ids):
        obj = cls(
            login=login,
            group_ids=group_ids
        )
        obj.save()

    @classmethod
    def get(cls, login):
        obj = cls.objects(login=login).first()
        return obj and list(obj.group_ids)


class StaffMembersCache(Document):
    """Persistent cache for staff group members."""
    group_id = StringField(primary_key=True)
    member_logins = ListField(StringField())

    @classmethod
    def set(cls, group_id, member_logins):
        obj = cls(
            group_id=str(group_id),
            member_logins=member_logins
        )
        obj.save()

    @classmethod
    def get(cls, group_id):
        obj = cls.objects(group_id=str(group_id)).first()
        return obj and list(obj.member_logins)


class AccessRight(Enum):
    Read = 'read'
    Write = 'write'


class Role(EmbeddedDocument):
    module = StringField(min_length=1, max_length=64, required=True)
    access_right = EnumField(AccessRight, required=True)


class User(Document):
    id = StringField(min_length=1, max_length=64, primary_key=True)
    roles = ListField(EmbeddedDocumentField(Role))


class Group(Document):
    id = StringField(min_length=1, max_length=64, primary_key=True)
    roles = ListField(EmbeddedDocumentField(Role))
