# coding: utf-8
import logging

import requests

from sepelib.yandex.staff import StaffClient, StaffError, get_group_ids, get_group_member_logins
from .dao import StaffGroupsCache, StaffMembersCache


class CachingStaffError(Exception):
    def __init__(self, error, cached):
        self.error = error
        self.cached = cached


class ICachingStaffClient(object):
    """
    Interface to be used in dependency injection.
    """


class CachingStaffClient(object):
    _USE_CACHE_ERRORS = (requests.HTTPError, requests.ConnectionError, requests.Timeout, StaffError)

    def __init__(self, staff_client):
        self._staff_client = staff_client
        self._log = logging.getLogger(__name__)

    @classmethod
    def from_config(cls, d):
        return cls(staff_client=StaffClient.from_config(d))

    def get_groups_ids(self, login):
        try:
            group_ids = get_group_ids(self._staff_client, login)
        except self._USE_CACHE_ERRORS as e:
            cached_group_ids = StaffGroupsCache.get(login)
            raise CachingStaffError(error=e, cached=cached_group_ids)
        else:
            StaffGroupsCache.set(login, group_ids)
            return group_ids

    def get_group_member_logins(self, group_id):
        try:
            member_logins = get_group_member_logins(self._staff_client, group_id)
        except self._USE_CACHE_ERRORS as e:
            member_logins = StaffMembersCache.get(group_id)
            raise CachingStaffError(error=e, cached=member_logins)
        else:
            StaffMembersCache.set(group_id, member_logins)
            return member_logins
