"""
Various mimetype utilities
"""
from collections import OrderedDict

from sepelib.core import config
from sepelib.core.exceptions import Error
from flask import request, json
import yaml
import msgpack
import six

__all__ = ['MimeException', 'MimeRenderException', 'MimeLoaderException',
           'mime_load', 'mime_render', 'choose_mime']


def request_wants_json():
    try:
        default_format = config.get_value('web.default_render_format', None)
    except Error:
        default_format = None

    # Useful for debugging from browser
    if request.args.get('fmt', default_format) == 'json':
        return True

    best = request.accept_mimetypes.best_match(['application/json', 'text/html', 'text/plain'])
    return best == 'application/json'


class MimeException(Exception):
    @property
    def reason(self):
        return self.args[0]


class MimeRenderException(MimeException):
    pass


class MimeLoaderException(MimeException):
    pass

# pretty clumsy mime renderer
# simplified version from flask-mimerenderer

# short names, used in ?fmt=<format>
XML = 'xml'
JSON = 'json'
TXT = 'txt'
HTML = 'html'
YAML = 'yaml'
MSGPACK = 'msgpack'


# ordered to make werkzeug's "best_match" work
# as intended (or at least this makes angularjs calls work)
_MIME_TYPES = OrderedDict([
    (JSON, ['application/json']),
    (MSGPACK, ['application/x-msgpack']),
    (TXT,  ['text/plain']),
    (YAML,  ['text/yaml',
             'text/x-yaml'
             'application/yaml',
             'application/x-yaml']),
    (HTML,  ['text/html']),
    (XML,  ['application/xml',
            'text/xml',
            'application/x-xml'])
])


def render_xml(obj):
    # it's a mockup, don't use it yet
    return "<result>{0}</result>".format(obj)


def render_txt(obj):
    # always try to return str instead of iterator
    # because that brings huge speed up
    # otherwise WSGI will call socket.send on every
    # yielded value and return response in chunk encoding

    # assume that it has only strings
    if isinstance(obj, list):
        return "\n".join(str(obj))
    if isinstance(obj, dict):
        body = "\t".join(
            "{0}:{1}".format(k, v) for k, v in six.iteritems(obj))
    elif hasattr(obj, '__iter__'):
        body = "\n".join(render_txt(i) for i in obj)
    elif obj is None:
        body = ''
    else:
        body = str(obj)
    return body


def render_json(obj):
    # browsers don't like non dictionary json responses
    # so to lower burden on controller -
    # make a dictionary automagically
    if not isinstance(obj, dict):
        obj = {'result': obj}
    return json.dumps(obj)


def render_html(obj):
    return "<html>{0}<html>".format(render_txt(obj))


def represent_odict(dump, tag, mapping, flow_style=None):
    """Like BaseRepresenter.represent_mapping, but does not issue the sort()."""
    # taken from http://blog.elsdoerfer.name/2012/07/26/make-pyyaml-output-an-ordereddict/
    value = []
    node = yaml.MappingNode(tag, value, flow_style=flow_style)
    if dump.alias_key is not None:
        dump.represented_objects[dump.alias_key] = node
    best_style = True
    if hasattr(mapping, 'items'):
        mapping = mapping.items()
    for item_key, item_value in mapping:
        node_key = dump.represent_data(item_key)
        node_value = dump.represent_data(item_value)
        if not (isinstance(node_key, yaml.ScalarNode) and not node_key.style):
            best_style = False
        if not (isinstance(node_value, yaml.ScalarNode) and not node_value.style):
            best_style = False
        value.append((node_key, node_value))
    if flow_style is None:
        if dump.default_flow_style is not None:
            node.flow_style = dump.default_flow_style
        else:
            node.flow_style = best_style
    return node


yaml.SafeDumper.add_representer(OrderedDict,
                                lambda dumper, value: represent_odict(dumper, u'tag:yaml.org,2002:map', value))


def render_yaml(obj):
    return yaml.safe_dump(obj, default_flow_style=False, allow_unicode=True)


def render_msgpack(obj):
    return msgpack.packb(obj)


# ==== loaders
# flask does it's best to parse request body
# into something meaningful
# so loader's argument is request
def load_text(request):
    """
    Create dictionary from line-separated key values.
    Beware: values are always strings
    """
    text = request.get_data() or request.form.keys[0]
    res = {}
    for line in text.splitlines():
        line = line.rstrip()
        parts = line.split(':', 1)
        if len(parts) != 2:
            raise ValueError("failed to split line: {0}".format(line))
        key, value = parts
        if not key:
            raise ValueError("empty key in line: {0}".format(line))
        if not value:
            raise ValueError("empty value in line: {0}".format(line))
        res[key] = value
    return res


def load_json(request):
    return request.json


def load_yaml(request):
    # Use yaml.safe_load cause yaml.load is insecure and may call any python function
    # http://pyyaml.org/wiki/PyYAMLDocumentation#LoadingYAML
    return yaml.safe_load(request.get_data() or request.form._keys[0])


def load_msgpack(request):
    return msgpack.unpackb(request.get_data() or request.form._keys[0])

# ====


_MIME_RENDERERS = {
    XML: render_xml,
    HTML: render_html,
    JSON: render_json,
    TXT: render_txt,
    YAML: render_yaml,
    MSGPACK: render_msgpack
}

_MIME_LOADERS = {
    XML: None,
    HTML: None,
    JSON: load_json,
    TXT: load_text,
    YAML: load_yaml,
    MSGPACK: load_msgpack
}


SUPPORTED_RENDERS = []  # supported str->object mimetypes
SUPPORTED_LOADERS = []  # supported object->str mimetypes

# mime-type to processor
RENDER_DICT = {}
LOADER_DICT = {}

for shortname, mimetypes in six.iteritems(_MIME_TYPES):
    for mime in mimetypes:
        render = _MIME_RENDERERS.get(shortname)
        if render is not None:
            RENDER_DICT[mime] = render
            SUPPORTED_RENDERS.append(mime)
        loader = _MIME_LOADERS.get(shortname)
        if loader is not None:
            LOADER_DICT[mime] = loader
            SUPPORTED_LOADERS.append(mime)


def choose_mime(fmt=None):
    # we can override document mime type with fmt argument
    fmt = request.args.get('fmt') or fmt
    if not (request.accept_mimetypes or fmt):
        # no types specified, choose default
        # it's documented in API, be careful if going to change
        fmt = config.get_value("web.default_render_format", "json")
    if fmt:
        types = _MIME_TYPES.get(fmt)
        mime = types[0] if types else None
    else:
        mime = request.accept_mimetypes.best_match(SUPPORTED_RENDERS)
    if not mime:
        raise MimeRenderException("No renderer for mime '{0}'".format(
            fmt or request.accept_mimetypes))
    return mime


def mime_render(document, fmt=None):
    mime = choose_mime(fmt)
    renderer = RENDER_DICT[mime]
    body = renderer(document)
    return body, mime


def mime_load(request, fmt=None):
    """
    :param fmt: override supplied request content-type
    :rtype: dict
    """
    # we can override document mime type with fmt argument
    fmt = fmt or request.args.get('fmt')
    if fmt:
        types = _MIME_TYPES.get(fmt)
        if not types:
            raise MimeLoaderException("Not supported fmt '{0}'".format(request.mimetype))
        mime = types[0] if types else None
    else:
        mime = request.mimetype
    loader = LOADER_DICT.get(mime)
    if loader is None:
        raise MimeLoaderException("Not supported content-type '{0}'".format(mime))
    return loader(request)
