"""
HTTP server boilerplate code.
"""

import copy
import datetime
import logging
import logging.handlers
import re
import six

import yaml
from werkzeug.middleware.proxy_fix import ProxyFix
from flask import request, abort, make_response, json, Response
from gevent import socket
from gevent import pywsgi

from sepelib.core import config
from sepelib.metrics.registry import metrics_inventory
from sepelib.util.fs import set_close_exec
from sepelib.util.log import reopen_logs, create_handler_from_config
from sepelib.flask.auth.util import login_exempt
from .h import prep_response
from .perfcounter import PerfCounter
from .statuscounter import StatusCounter

log = logging.getLogger('flask')


def setup_access_log_stream(cfg):
    """
    Abuse handy log handler and
    create stream like object for wsgi to write access log.
    """

    class StreamToHandler(object):
        """
        Fake file-like stream object that redirects writes to a loghander instance.
        """

        def __init__(self, handler):
            self.handler = handler
            self.linebuf = ''

        def isatty(self):
            return False

        def write(self, buf):
            # remove line endings, handler will add them anyway
            buf = buf.rstrip()
            record = logging.makeLogRecord({'msg': buf,
                                            'levelno': logging.DEBUG})
            self.handler.handle(record)

    # to support old configs when cfg is a string
    if isinstance(cfg, six.string_types):
        handler = logging.handlers.WatchedFileHandler(cfg)
    else:
        handler = create_handler_from_config(cfg)

    formatter = logging.Formatter("%(message)s")
    handler.setLevel(logging.DEBUG)
    handler.setFormatter(formatter)

    stream = StreamToHandler(handler)
    return stream


class ReverseProxied(object):
    """
    Wrap the application in this middleware and configure the front-end server
    to add these headers, to let you quietly bind this to a URL other
    than and to an HTTP scheme that is different than what is used locally.

    In nginx:
        location /myprefix {
            proxy_pass http://192.168.0.1:5001;     # where Flask app runs
            proxy_set_header Host $host;
            proxy_set_header X-Forwarded-For $proxy_add_x_forwarded_for;
            proxy_set_header X-Scheme $scheme;
            proxy_set_header X-Script-Name /myprefix;
            }

    :param app: the WSGI application
    """

    def __init__(self, app):
        self.app = app

    def __call__(self, environ, start_response):
        script_name = environ.get('HTTP_X_SCRIPT_NAME', '')
        if script_name:
            environ['SCRIPT_NAME'] = script_name
            path_info = environ['PATH_INFO']
            if path_info.startswith(script_name):
                environ['PATH_INFO'] = path_info[len(script_name):]
        scheme = environ.get('HTTP_X_SCHEME', '')
        if scheme:
            environ['wsgi.url_scheme'] = scheme
        return self.app(environ, start_response)


class WSGIHandler(pywsgi.WSGIHandler):
    def format_request(self):
        """
        Overridden to try to log client address from headers.
        """
        now = datetime.datetime.now().replace(microsecond=0)
        length = self.response_length or '-'
        if self.time_finish:
            delta = '%.6f' % (self.time_finish - self.time_start)
        else:
            delta = '-'
        if 'HTTP_X_FORWARDED_FOR' in self.environ:
            # Good case: we have X-Forwarded-For
            addr = self.environ['HTTP_X_FORWARDED_FOR'].split(',')
            client_address = addr[0].strip()
        else:
            client_address = self.client_address[0] if isinstance(self.client_address, tuple) else self.client_address
        return '%s - - [%s] "%s" %s %s %s' % (
            client_address or '-',
            now,
            getattr(self, 'requestline', ''),
            (getattr(self, 'status', None) or '000').split()[0],
            length,
            delta)


class WebServer(object):
    """
    Object which:
      * encapsulates flask server (probably more than one)
    Uses gevent python wsgi implementation because in 1.0
    gevent abandons libevent and it's http server implementation.
    """
    _DEFAULT_BACKLOG = 1024

    @classmethod
    def _create_server_socket(cls, host, port, backlog=None):
        """
        Try to create ipv6 socket, falling back to ipv4.
        Start listening on it. :return: listening socket
        """
        if host == '0.0.0.0':
            host = ''
        try:
            listener = socket.socket(socket.AF_INET6)
        except EnvironmentError as e:
            log.warn("failed to create ipv6 socket: {0}".format(e.strerror))
            log.warn("falling back to ipv4 only")
            listener = socket.socket()
        listener.setsockopt(socket.SOL_SOCKET, socket.SO_REUSEADDR, 1)
        # mark the socket fd as non-inheritable
        set_close_exec(listener.fileno())
        try:
            listener.bind((host, port))
        except socket.error:
            # it's useful to log what host and port we failed to bind
            log.error("failed to bind '{}' on port {}".format(host, port))
            raise
        listener.listen(backlog if backlog is not None else cls._DEFAULT_BACKLOG)
        return listener

    @staticmethod
    def _wrap_app(app, proxies_num, proxyfix=False):
        if proxies_num:
            app.wsgi_app = ReverseProxied(app.wsgi_app)
            if proxyfix:
                app.wsgi_app = ProxyFix(app.wsgi_app)
        return app

    def __init__(self, cfg, app, version, logstream=None, proxyfix=False):
        self.cfg = cfg
        self.version = version

        # === setup http ===
        webcfg = cfg['web']
        http_cfg = webcfg['http']
        self.app = self._wrap_app(app, http_cfg.get('proxies_num', 1), proxyfix=proxyfix)
        host, port = http_cfg['host'], http_cfg['port']
        if webcfg.get('access_log'):
            logstream = setup_access_log_stream(webcfg['access_log'])
        listener = self._create_server_socket(host, port, backlog=http_cfg.get('backlog'))
        self.wsgi = pywsgi.WSGIServer(listener,
                                      application=self.app,
                                      log=logstream,
                                      handler_class=WSGIHandler)

        # add some performance hooks
        self.perf = PerfCounter()
        self.perf.augment(self.app)
        StatusCounter().augment(self.app)
        self._register_urls()

    def run(self):
        log.info("starting flask interface on: '{0[0]}':{0[1]}".format(self.wsgi.address))
        self.wsgi.serve_forever()

    def stop(self):
        log.info('stopping flask service')
        self.wsgi.stop(timeout=1)

    def _register_urls(self):
        # functions part
        self.app.add_url_rule('/ping', view_func=self.ping)
        self.app.add_url_rule('/version', view_func=self.render_version)
        self.app.add_url_rule('/config', view_func=self.render_config)
        self.app.add_url_rule('/reopen_log', view_func=self.reopen_log)
        self.app.add_url_rule('/metrics/', view_func=self.render_metrics)
        self.app.add_url_rule('/metrics/<path:path>', view_func=self.render_metrics)
        self.app.add_url_rule('/yasm_stats/', view_func=self.render_yasm_stats)

    @classmethod
    @login_exempt
    def ping(cls):
        return Response('', status=200)

    @staticmethod
    @login_exempt
    def render_metrics(path='/'):
        d = {}
        # path should start with '/'
        if not path.startswith('/'):
            path = '/{}'.format(path)
        for key, metrics in metrics_inventory.items(path):
            # we want to print only those suffixes, not whole keys
            # if path == '/services', we want to print
            # /http
            # /thrift
            if path != '/':
                key = key.split('/')[-1]
            d[key] = metrics.dump_metrics()
        return prep_response(d)

    @staticmethod
    @login_exempt
    def render_yasm_stats(sub_re=re.compile(r'[<>_/\.:]')):
        """
        Returns a list with stats for yasm format.
        **Very experimental feature**
        """
        stats_result = []
        for key, metrics_registry in metrics_inventory.items('/'):
            # Metrics name must match ^[a-zA-Z0-9\-]
            # All keys start with '/' and we do not want all metrics start with '-'
            for subkey, metrics in metrics_registry.dump_yasm_metrics():
                prefix = sub_re.sub('-', key[1:] + '-' + subkey)
                for suffix, value in metrics.items():
                    name = '{}-{}'.format(prefix, suffix)
                    stats_result.append((name, value))
        response = make_response(json.dumps(stats_result))
        response.headers['Content-Type'] = 'application/json'
        return response

    @login_exempt
    def render_version(self):
        return prep_response({'version': self.version}, fmt='txt')

    @login_exempt
    def render_config(self):
        output_config = self.cfg
        hidden_options = config.get_value("web.http.hidden_config_options", default=None, config=self.cfg)

        if hidden_options:
            undefined = object()
            output_config = copy.deepcopy(output_config)

            for hidden_option in hidden_options:
                if config.get_value(hidden_option, default=undefined, config=output_config) is not undefined:
                    config.set_value(hidden_option, "HIDDEN", config=output_config)

        obj = yaml.safe_dump(output_config, default_flow_style=False)
        return prep_response(obj, fmt='txt')

    @login_exempt
    def reopen_log(self):
        """
        Check if request came from localhost and try to reopen log file(s).
        """
        remote_addr = request.remote_addr
        log.info("received log reopen request from '{0}'".format(remote_addr))
        if remote_addr in ('127.0.0.1', '::1', '::ffff:127.0.0.1'):
            reopen_logs()
            return prep_response({'result': 'OK'}, fmt='txt')
        else:
            abort(403)
