import os
import fcntl
import errno
import logging
import threading
from gevent import socket, spawn
from gevent.event import AsyncResult
from six.moves.queue import Queue

from sepelib.util.fs.util import close_ignore


__all__ = ['ThreadMixin']


class ThreadMixin(threading.Thread):
    """
    Mixin object which allows to defer function
    to thread and put calling greenlet to sleep waiting for
    AsyncResult.

    Example usage.
    Suppose you have class which performs a job synchronously:
    class CacheBuilder(object):
        def build_cache(self, input):
            result = {}
            for i in input:
                result[i] = long_computation(i)
            return result

    And you want to process this asynchronously in your gevent based application.
    This is how it can be achieved:
    class AsyncCacheBuilder(CacheBuilder, ThreadMixin):
        def build_cache_async(self, input):
            return self._put(self.build_cache, input)

    Now you have an async version of your method and free to choose.
    """
    MAX_QUEUE_LEN = 100

    def __init__(self, name):
        threading.Thread.__init__(self, name=name)
        self.log = logging.getLogger(name)
        self.daemon = True
        self.running = False
        self.read_end, self.write_end = os.pipe()
        fcntl.fcntl(self.read_end, fcntl.F_SETFL, os.O_NONBLOCK)
        fcntl.fcntl(self.write_end, fcntl.F_SETFL, os.O_NONBLOCK)
        self.queue = Queue(self.MAX_QUEUE_LEN)
        self._wakeupper = spawn(self._wakeup_loop)

    def _wakeup_loop(self):
        """
        Wait for read from pipe to wake up main thread.
        Causes automagic pending events (created from thread) dispatching.
        """
        while 1:
            socket.wait_read(self.read_end)
            try:
                data = os.read(self.read_end, 4096)
            except EnvironmentError as e:
                self.log.error("failed to read event from pipe: {0}".format(e.strerror))
                break
            if not data:
                break

    def _put(self, func, *args, **kw):
        """
        Defer :param func: to thread.
        Use this method to call your public long running
        methods in thread.
        """
        res = AsyncResult()
        self.queue.put_nowait((func, args, kw, res))
        return res.get()

    def run(self):
        self.running = True
        while 1:
            task = self.queue.get()
            if task is None:
                break
            func, args, kw, result = task
            try:
                res = func(*args, **kw)
            except Exception as e:
                result.set_exception(e)
            else:
                result.set(res)
            finally:
                self.queue.task_done()
                while 1:
                    try:
                        os.write(self.write_end, '0')
                    except EnvironmentError as e:
                        # it's ok if pipe is full
                        # as soon as read_end will wake up - it would
                        # dispatch all pending events
                        if e.errno == errno.EBADF:  # fd is closed
                            break
                        elif e.errno in [errno.EAGAIN, errno.EWOULDBLOCK]:
                            pass
                        elif e.errno == errno.EINTR:
                            continue
                        elif self.running:
                            self.log.error("failed to write event to pipe: {0}".format(
                                e.strerror))
                    break

    def stop(self):
        self.running = False
        self.queue.put_nowait(None)
        if self.read_end != -1:
            close_ignore(self.read_end)
            self.read_end = -1
        if self.write_end != -1:
            close_ignore(self.write_end)
            self.write_end = -1
        if self._wakeupper is not None:
            self._wakeupper.kill()
            self._wakeupper = None
