import re
import time

from .meters import Counter, Histogram, Meter, Timer
from .util.trie import Trie, Path


class MetricsRegistry(object):
    """
    A single interface used to gather metrics on a service. It keeps track of
    all the relevant Counters, Meters, Histograms, and Timers. It does not have
    a reference back to its service. The service would create a
    ``MetricsRegistry`` to manage all of its metrics tools.
    """
    def __init__(self, clock=time):
        self._timers = {}
        self._meters = {}
        self._counters = {}
        self._histograms = {}
        self._gauges = {}
        self._clock = clock

    def counter(self, key):
        """
        Gets a counter based on a key, creates a new one if it does not exist.

        :param key: name of the metric
        :type key: str

        :return: Counter
        """
        if key not in self._counters:
            self._counters[key] = Counter()
        return self._counters[key]

    def histogram(self, key):
        """
        Gets a histogram based on a key, creates a new one if it does not exist.

        :param key: name of the metric
        :type key: str

        :return: L{Histogram}
        """
        if key not in self._histograms:
            self._histograms[key] = Histogram(clock=self._clock)
        return self._histograms[key]

    def gauge(self, key, gauge=None):
        if key not in self._gauges:
            self._gauges[key] = gauge
        return self._gauges[key]

    def meter(self, key):
        """
        Gets a meter based on a key, creates a new one if it does not exist.

        :param key: name of the metric
        :type key: str

        :rtype: Meter
        """
        if key not in self._meters:
            self._meters[key] = Meter(clock=self._clock)
        return self._meters[key]

    def timer(self, key):
        """
        Gets a timer based on a key, creates a new one if it does not exist.

        :param key: name of the metric
        :type key: str

        :rtype: Timer
        """
        if key not in self._timers:
            self._timers[key] = Timer(clock=self._clock)
        return self._timers[key]

    def clear(self):
        self._meters.clear()
        self._counters.clear()
        self._gauges.clear()
        self._timers.clear()
        self._histograms.clear()

    def _get_counter_metrics(self, key):
        if key in self._counters:
            counter = self._counters[key]
            return {"count": counter.get_count(), "uuid": counter.uuid}
        return {}

    def _get_histogram_metrics(self, key):
        if key in self._histograms:
            histogram = self._histograms[key]
            snapshot = histogram.get_snapshot()
            res = {"avg": histogram.get_mean(),
                   "count": histogram.get_count(),
                   "max": histogram.get_max(),
                   "min": histogram.get_min(),
                   "std_dev": histogram.get_stddev(),
                   "75_percentile": snapshot.get_75th_percentile(),
                   "95_percentile": snapshot.get_95th_percentile(),
                   "99_percentile": snapshot.get_99th_percentile(),
                   "999_percentile": snapshot.get_999th_percentile(),
                   "uuid": histogram.uuid
                   }
            return res
        return {}

    def _get_meter_metrics(self, key):
        if key in self._meters:
            meter = self._meters[key]
            res = {"15m_rate": meter.get_fifteen_minute_rate(),
                   "5m_rate": meter.get_five_minute_rate(),
                   "1m_rate": meter.get_one_minute_rate(),
                   "mean_rate": meter.get_mean_rate(),
                   "uuid": meter.uuid
                   }
            return res
        return {}

    def _get_timer_metrics(self, key):
        if key in self._timers:
            timer = self._timers[key]
            snapshot = timer.get_snapshot()
            res = {"avg": timer.get_mean(),
                   "count": timer.get_count(),
                   "max": timer.get_max(),
                   "min": timer.get_min(),
                   "std_dev": timer.get_stddev(),
                   "75_percentile": snapshot.get_75th_percentile(),
                   "95_percentile": snapshot.get_95th_percentile(),
                   "99_percentile": snapshot.get_99th_percentile(),
                   "999_percentile": snapshot.get_999th_percentile(),
                   "uuid": timer.uuid
                   }
            return res
        return {}

    def get_metrics(self, key):
        """
        Gets all the metrics for a specified key.

        :param key: name of the metric
        :type key: C{str}

        :return: C{dict}
        """
        metrics = {}
        for getter in (self._get_counter_metrics, self._get_histogram_metrics,
                       self._get_meter_metrics, self._get_timer_metrics):
            metrics.update(getter(key))
        return metrics

    def get_yasm_format_metrics(self, key):
        metrics = {}
        for source in (self._counters, self._histograms,
                       self._meters, self._timers):
            meter = source.get(key)
            if meter is not None:
                metrics.update(meter.get_yasm_format_metrics())

        return metrics

    def dump_metrics(self):
        """
        Formats all of the metrics and returns them as a dict.

        :return: list of dict
        """
        metrics = {}
        for metric_type in (self._counters,
                            self._histograms,
                            self._meters,
                            self._timers):
            for key in metric_type.keys():
                metrics[key] = self.get_metrics(key)
        return metrics

    def dump_yasm_metrics(self):
        keys = set()
        for metric_type in (self._counters,
                            self._histograms,
                            self._meters,
                            self._timers):
            keys.update(metric_type.keys())
        for key in keys:
            yield key, self.get_yasm_format_metrics(key)


class RegexRegistry(MetricsRegistry):
    """
    A single interface used to gather metrics on a service. This class uses a regex to combine
    measures that match a pattern. For example, if you have a REST API, instead of defining
    a timer for each method, you can use a regex to capture all API calls and group them.
    A pattern like '^/api/(?P<model>)/(?P<verb>)?$' will group and measure the following:
        /api/users/1 -> users
        /api/users/1/edit -> users/edit
        /api/users/2/edit -> users/edit
    """
    def __init__(self, pattern=None, clock=time):
        super(RegexRegistry, self).__init__(clock)
        if pattern is not None:
            self.pattern = re.compile(pattern)
        else:
            self.pattern = re.compile('^$')

    def _get_key(self, key):
        matches = self.pattern.finditer(key)
        key = '/'.join((v for match in matches for v in match.groups() if v))
        return key

    def timer(self, key):
        return super(RegexRegistry, self).timer(self._get_key(key))

    def histogram(self, key):
        return super(RegexRegistry, self).histogram(self._get_key(key))

    def counter(self, key):
        return super(RegexRegistry, self).counter(self._get_key(key))

    def gauge(self, key, gauge=None):
        return super(RegexRegistry, self).gauge(self._get_key(key), gauge)

    def meter(self, key):
        return super(RegexRegistry, self).meter(self._get_key(key))


_global_registry = MetricsRegistry()


def counter(key):
    return _global_registry.counter(key)


def histogram(key):
    return _global_registry.histogram(key)


def meter(key):
    return _global_registry.meter(key)


def timer(key):
    return _global_registry.timer(key)


def dump_metrics():
    return _global_registry.dump_metrics()


def clear():
    return _global_registry.clear()


class MetricsInventory(object):
    class _PathTrie(Trie):
        KeyFactory = '/'.join

    def __init__(self):
        self._trie = self._PathTrie()

    def iter_prefixes(self, key):
        return self._trie.iter_prefixes(Path(key))

    def iter_prefix_values(self, key):
        return self._trie.iter_prefix_values(Path(key))

    def keys(self, prefix=None):
        return self._trie.keys(prefix=Path(prefix) if prefix else None)

    def values(self, prefix=None):
        return self._trie.values(prefix=Path(prefix) if prefix else None)

    def items(self, prefix=None):
        return self._trie.items(prefix=Path(prefix) if prefix else None)

    def children(self, key):
        return self._trie.children(Path(key))

    def __contains__(self, item):
        return Path(item) in self._trie

    def get_metrics(self, key):
        key = Path(key)
        registry = self._trie.get(key)
        if registry is None:
            registry = MetricsRegistry()
            self._trie[key] = registry
        return registry

    def clear(self):
        self._trie = self._PathTrie()


metrics_inventory = MetricsInventory()


def count_calls(fn):
    """
    Decorator to track the number of times a function is called.

    :param fn: the function to be decorated
    :type fn: func

    :return: the decorated function
    :rtype: func
    """
    def wrapper(*args, **kwargs):
        counter("%s_calls" % fn.__name__).inc()
        return fn(*args, **kwargs)
    return wrapper


def meter_calls(fn):
    """
    Decorator to the rate at which a function is called.

    :param fn: the function to be decorated
    :type fn: func

    :return: the decorated function
    :rtype: func
    """
    def wrapper(*args, **kwargs):
        meter("%s_calls" % fn.__name__).mark()
        return fn(*args, **kwargs)
    return wrapper


def hist_calls(fn):
    """
    Decorator to check the distribution of return values of a function.

    :param fn: the function to be decorated
    :type fn: func

    :return: the decorated function
    :rtype: func
    """
    def wrapper(*args, **kwargs):
        _histogram = histogram("%s_calls" % fn.__name__)
        rtn = fn(*args, **kwargs)
        if isinstance(rtn, (int, float)):
            _histogram.update(rtn)
        return rtn
    return wrapper


def time_calls(fn):
    """
    Decorator to time the execution of the function.

    :param fn: the function to be decorated
    :type fn: C{func}

    :return: the decorated function
    :rtype: func
    """
    def wrapper(*args, **kwargs):
        _timer = timer("%s_calls" % fn.__name__)
        with _timer.time():
            return fn(*args, **kwargs)
    return wrapper
