import math
import six


class Snapshot(object):
    MEDIAN = 0.5
    P75_Q = 0.75
    P95_Q = 0.95
    P99_Q = 0.99
    P999_Q = 0.999
    BASE = 1.5

    def __init__(self, values):
        super(Snapshot, self).__init__()
        self.values = sorted(values)

    def get_size(self):
        return len(self.values)

    def get_median(self):
        return self.get_percentile(Snapshot.MEDIAN)

    def get_75th_percentile(self):
        return self.get_percentile(Snapshot.P75_Q)

    def get_95th_percentile(self):
        return self.get_percentile(Snapshot.P95_Q)

    def get_99th_percentile(self):
        return self.get_percentile(Snapshot.P99_Q)

    def get_999th_percentile(self):
        return self.get_percentile(Snapshot.P999_Q)

    def get_percentile(self, percentile):
        if percentile < 0 or percentile > 1:
            raise ValueError("{} is not in [0..1]".format(percentile))
        length = len(self.values)
        if length == 0:
            return 0
        pos = percentile * (length + 1)
        if pos < 1:
            return self.values[0]
        if pos >= length:
            return self.values[-1]
        lower = self.values[int(pos) - 1]
        upper = self.values[int(pos)]
        return lower + (pos - int(pos)) * (upper - lower)

    def get_histogram(self, size=50):
        """
        Returns histogram as list of (left border, weight).
        Implementation matching that of golovan united stats handler.

        Number of buckets is variable and limited by the size argument.
        Buckets are statically allocated powers of 1.5 and zero if required.
        """
        if size < 1:
            raise ValueError('Wrong size value {}, must be greater than 1'.format(size))

        nonzero_values = list(filter(None, self.values))
        zeroes_count = len(self.values) - len(nonzero_values)
        if not nonzero_values:
            return[[0, len(self.values)]]

        max_idx = int(math.floor(math.log(nonzero_values[-1], self.BASE)))
        if zeroes_count > 0:
            min_idx = max_idx - size
        else:
            min_idx = int(math.floor(math.log(nonzero_values[0], self.BASE)))

        min_idx = max(min_idx, max_idx - size + 1)

        buckets = [[self.BASE**i, 0] for i in six.moves.xrange(min_idx, max_idx+1)]
        # smallest bucket is always the minimum value
        buckets[0][0] = self.values[0]
        for i in self.values:
            idx = max(0, int(math.floor(math.log(i, self.BASE)) - min_idx)) if i else 0
            buckets[idx][1] += 1

        return buckets
