"""Provides an extended Document class which is intended to be used as base class for all MongoEngine models."""

from __future__ import unicode_literals

import mongoengine

from sepelib.gevent.utils import gevent_idle_iter
from sepelib.mongo.util import IS_PYMONGO_2
from sepelib.util.misc import doesnt_override


class QuerySet(mongoengine.QuerySetNoCache):
    """A custom QuerySet which is used as default for all MongoEngine models.

    The default QuerySet (mongoengine.QuerySet) caches all objects read from cursor! This is why our QuerySet is derived
    from QuerySetNoCache.
    """

    @doesnt_override(mongoengine.QuerySetNoCache)
    def get_ids(self):
        return [obj.pk for obj in gevent_idle_iter(self.only("pk"))]


class Document(mongoengine.Document):
    """Base class for all MongoEngine models."""

    meta = {
        "abstract": True,

        # Don't create indexes automatically.
        # All indexes must be created by manual call of sepelib.mongo.util.ensure_all_indexes() on application startup.
        "auto_create_index": False,

        # Customize default QuerySet
        "queryset_class": QuerySet,
    }

    @classmethod
    @doesnt_override(mongoengine.Document)
    def get_collection(cls, read_preference=None):
        # Always get the collection using MongoEngine's method first - it does some collection prepare actions.
        coll = cls._get_collection()
        if read_preference is None:
            return coll

        if not IS_PYMONGO_2:
            return cls._get_db().get_collection(cls._get_collection_name(), read_preference=read_preference)

        coll = cls._get_db()[cls._get_collection_name()]
        coll.read_preference = read_preference
        return coll

    @doesnt_override(mongoengine.Document)
    def copy(self):
        """Returns a deep copy of the document."""

        return self._from_son(self.to_mongo())


def is_equal(a, b):
    """Compares two documents."""

    if a is b:
        return True
    elif a is None or b is None:
        return False
    else:
        return a.to_mongo() == b.to_mongo()
