from mongoengine.fields import BaseField, ListField

from . import util
import six


class EnumField(BaseField):
    """Enumerator field.

    :type enum: :class:`enum.Enum`
    """
    def __init__(self, enum, *args, **kwargs):
        self.enum = enum
        kwargs['choices'] = [choice for choice in enum]
        super(EnumField, self).__init__(*args, **kwargs)

    def to_python(self, value):
        """Convert a MongoDB-compatible type to a Python type."""
        if not isinstance(value, (six.string_types, six.text_type)) and value in self.enum:
            return value
        return self.enum[value]

    def to_mongo(self, enum):
        """Convert a Python type to a MongoDB-compatible type."""
        return enum.name


class SetField(ListField):
    """ Set field.

        Extends ListField, so that's how it's represented in Mongo.
    """
    def __init__(self, *args, **kwargs):
        if not util.is_patched():
            self.error('MongoEngine must be patched for this field to work. '
                       'see `.mongo.util.patch` function')
        super(SetField, self).__init__(*args, **kwargs)

    def __set__(self, instance, value):
        if value:
            value = set(value)
        return super(SetField, self).__set__(instance, value)

    def to_mongo(self, value):
        if value is not None:
            value = set(value)
        return super(SetField, self).to_mongo(value)

    def to_python(self, value):
        retval = super(SetField, self).to_python(value)
        return set(retval) if retval is not None else retval

    def validate(self, value):
        if value is not None and not isinstance(value, set):
            self.error('Only sets may be used.')
