import logging
import pymongo

from pymongo import errors, MongoClient, DESCENDING, ASCENDING
from pymongo.read_preferences import ReadPreference
from sepelib.mongo.util import IS_PYMONGO_2


__all__ = ['MongoError',
           'AutoReconnect', 'MongoStorage', 'ConnectionFailure',
           'ReadPreference', 'DESCENDING', 'ASCENDING']


log = logging.getLogger('mongo')


MongoError = errors.PyMongoError
DuplicateKeyError = errors.DuplicateKeyError
AutoReconnect = errors.AutoReconnect
ConnectionFailure = errors.ConnectionFailure


class MongoStorage(object):
    """
    Simple mongodb wrapper.
    """
    # we use not so big timeout, because it's time before giving up
    # each replica. no need to wait forever for each - let's quickly
    # scan through them
    CONNECT_TIMEOUT = 5
    NETWORK_TIMEOUT = 60
    # for w_option - check out replica set options
    # in actual mongodb rs.config()

    def __init__(self, cfg):
        self.addr = cfg['addr']
        self.dbname = cfg['dbname']
        self.login = cfg.get('login')
        self.password = cfg.get('password')
        # now we can construct URI, it's the only way to set login/password
        if self.login and self.password:
            self.uri = 'mongodb://{login}:{password}@{addr}/{database}'.format(
                login=self.login,
                password=self.password,
                addr=self.addr,
                database=self.dbname
            )
        else:
            self.uri = 'mongodb://{addr}/{database}'.format(
                addr=self.addr,
                database=self.dbname
            )
        # use 'get' to be able to work without replica setup
        # in test environment
        self.replicaset = cfg.get('replicaset')
        self.read_preference = getattr(
            ReadPreference, cfg.get('read_preference', 'SECONDARY_PREFERRED'))
        self.w_option = cfg.get('w_option')
        self.fsync = cfg.get('fsync', False)
        self.sock_timeout = cfg.get('socktimeout', self.NETWORK_TIMEOUT) * 1000
        self._conn = None
        self.db = None

    def _connection_options(self):
        options = {
            'host': self.uri,
            'fsync': self.fsync,
            'connect': False,
            'connectTimeoutMS': self.CONNECT_TIMEOUT * 1000,
            'socketTimeoutMS': self.sock_timeout
        }
        if IS_PYMONGO_2:
            options['use_greenlets'] = True
        conn_cls = MongoClient
        if self.replicaset:
            options['replicaSet'] = self.replicaset
            options['read_preference'] = self.read_preference
            if self.w_option:
                options['w'] = self.w_option
            if IS_PYMONGO_2:
                conn_cls = pymongo.MongoReplicaSetClient
        return options, conn_cls

    def make_connection(self):
        options, conn_cls = self._connection_options()
        conn = conn_cls(**options)
        # Sync wait of connect
        # https://api.mongodb.com/python/current/migrate-to-pymongo3.html#mongoclient-connects-asynchronously
        conn.admin.command("ismaster")
        host, port = (conn.host, conn.port) if IS_PYMONGO_2 else conn.address
        log.info("created connection to {0}:{1}".format(host, port))
        return conn

    def __getattr__(self, item):
        self.start()
        return getattr(self.db, item)

    __getitem__ = __getattr__

    @property
    def status(self):
        if not self._conn:
            return 'FAIL'
        last_error = self._conn.admin.command({'getLastError': 1})
        return last_error['err'] or 'OK'

    @property
    def replica_status(self):
        self.start()
        return self._conn.admin.command({'replSetGetStatus': 1})

    @property
    def collection_status(self):
        if not self._conn:
            self.start()
        stat = []
        for name in self.db.collection_names():
            if name == 'system.indexes':
                continue
            s = self.db.command({'collStats': name})
            s['name'] = name
            stat.append(s)
        stat.sort(key=lambda k: k['name'])
        return stat

    @property
    def collection_names(self):
        if not self._conn:
            self.start()
        return self.db.collection_names()

    def start(self):
        if not self._conn:
            self._conn = self.make_connection()
            self.db = self._conn[self.dbname]

    def stop(self):
        if self._conn is not None:
            self.db = None
            self._conn.close()
            self._conn = None
