# coding: utf-8
import os.path
import logging
import logging.handlers


__all__ = ['ReopenFileHandler', 'create_handler_from_config']


class ReopenFileHandler(logging.StreamHandler):
    """File handler which supports reopening of logs.

    Re-opening should be used instead of the 'rollover' feature of
    the FileHandler from the standard library's logging package.
    """

    def __init__(self, filename, mode="a"):
        self.filename = os.path.abspath(filename)
        self.mode = mode
        logging.StreamHandler.__init__(self, open(self.filename, self.mode))

    def close(self):
        self.stream.close()

    def reopen(self):
        self.close()
        self.stream = open(self.filename, self.mode)


def create_handler_from_config(cfg, default_handler_class=logging.handlers.WatchedFileHandler):
    """Creates a logging handler from a config.
    Config must look as follows:
    {
        'handler_class': 'TimedRotatingFileHandler',
        'params': {
            'when': 'midnight'
            'backupCount': 10
        },
        'filepath': '/nanny/nanny_extended_access.log'
    }
    """
    handler_name = cfg.get('handler_class')
    if handler_name:
        if handler_name == 'ReopenFileHandler':
            handler_class = ReopenFileHandler
        else:  # look in standard library
            handler_class = getattr(logging.handlers, handler_name, None)
            if not handler_class:
                raise ValueError("Unknown handler class '{}' provided in config".format(handler_name))
    else:
        handler_class = default_handler_class
    kwargs = cfg.get('params', {})
    # to be compatible with earlier configs we assume that
    # first argument is always a file path
    handler = handler_class(cfg['filepath'], **kwargs)
    return handler
