import types
import os
import sys
import socket
import logging
import logging.handlers
import six


try:
    from clint.textui import colored
except ImportError:
    # no colors in console
    colored = None

from sepelib.util.exc import format_exc


__all__ = ['setup_logging_to_stdout', 'setup_logging_to_file', 'reopen_logs']


# Patch logging to have getChild function from Python 2.7.x
if not hasattr(logging.Logger, 'getChild'):
    def _getChild(self, suffix):
        if self.root is not self:
            suffix = '.'.join((self.name, suffix))
        return self.manager.getLogger(suffix)

    logging.Logger.getChild = _getChild


def _get_short_hostname():
    hostname = socket.gethostname()
    pos = hostname.find('.')
    if pos != -1:
        hostname = hostname[:pos]
    return hostname


class AugmentedFormatter(logging.Formatter):
    """
    Formatter that augments record instances with useful attributes
    """
    hostname = _get_short_hostname()

    def format(self, record):
        record.hostname = self.hostname
        return logging.Formatter.format(self, record)


if colored is not None:
    # colorful console is available
    # set appropriate formatter class
    class ColorfulFormatter(AugmentedFormatter):

        COLORS = {logging.ERROR: colored.red,
                  logging.WARN: colored.yellow}

        def format(self, record):
            color = self.COLORS.get(record.levelno)
            if color is not None:
                record.levelname = color(record.levelname)
            return AugmentedFormatter.format(self, record)

    Formatter = ColorfulFormatter
else:
    Formatter = AugmentedFormatter


class StreamToLogger(object):
    """
    Fake file-like stream object that redirects writes to a logger instance.
    From http://clck.ru/1DMQD.
    """
    def __init__(self, logger, log_level=logging.INFO):
        self.logger = logger
        self.log_level = log_level
        self.linebuf = ''

    def isatty(self):
        return False

    def write(self, buf):
        for line in buf.rstrip().splitlines():
            self.logger.log(self.log_level, line.rstrip())


# * prepend with host name
# * use asctime without backslashes to be able to grep without escaping
FORMAT = "%(hostname)s %(process)s %(asctime)s %(levelname)s [%(name)s] %(message)s"


def rename_loglevels():
    """
    Rename log levels so that all have the same length of 5 characters.
    This makes them look nice and consistent in log file.
    """
    short_levelnames = {logging.WARN: "WARN ",
                        logging.INFO: "INFO "}
    for k, v in six.iteritems(short_levelnames):
        logging.addLevelName(k, v)


def setup_logging_to_stdout():
    """
    Setup logging handler pushing everything into stdout.
    :rtype: logging.Handler
    """
    rename_loglevels()
    formatter = Formatter(FORMAT)

    logger = logging.getLogger()
    logger.setLevel(logging.DEBUG)

    stdout_handler = logging.StreamHandler(sys.stdout)
    stdout_handler.setLevel(logging.DEBUG)
    stdout_handler.setFormatter(formatter)
    logger.addHandler(stdout_handler)

    return stdout_handler


def redirect_std_fds(fd, redirect_stdout=True, redirect_stderr=True):
    """
    Redirect standart in/out descriptors into specified :param fd:
    This will help if someone attempts to write around sys.std{out,err} file objects.

    :type fd: int or fileobject
    :param redirect_stdout: do redirect stdout to fd
    :type redirect_stdout: bool
    :param redirect_stderr: do redirect stderr to fd
    :type redirect_stderr: bool
    """
    if not isinstance(fd, int):
        fd = fd.fileno()
    if redirect_stdout:
        os.dup2(fd, 1)
    if redirect_stderr:
        os.dup2(fd, 2)


def setup_logging_to_file(handler, redirect_stdout=True, redirect_stderr=True, logger_name=None, fmt=FORMAT):
    """
    :param handler: log handler to configure
    :type handler: logging.handlers.Handler
    :param redirect_stdout: do redirect stdout to log file
    :type redirect_stdout: bool
    :param redirect_stderr: do redirect stderr to log file
    :type redirect_stderr: bool
    :param logger_name: specified logger name
    :type logger_name: unicode | str
    :param fmt: log message format string
    :type fmt: unicode | str
    :return: configured logger
    :rtype: logging.Logger
    """
    rename_loglevels()
    logger = logging.getLogger(logger_name)
    logger.setLevel(logging.DEBUG)

    handler.setLevel(logging.DEBUG)

    formatter = Formatter(fmt)
    handler.setFormatter(formatter)
    logger.addHandler(handler)

    with open(os.devnull) as devnull:
        os.dup2(devnull.fileno(), 0)

    # task: we want to redirect std{out,err} to file
    # and *after* file rotation we want to redirect them again
    # it's not that simple
    # and a bit ugly - but thus we can use any handler that writes to file
    if isinstance(handler, logging.FileHandler):
        def _open(self):
            stream = self.__class__._open(self)
            redirect_std_fds(stream, redirect_stdout, redirect_stderr)
            return stream
        handler._open = types.MethodType(_open, handler)
        if handler.stream is not None:  # file is already opened - redirect manually
            redirect_std_fds(handler.stream, redirect_stdout, redirect_stderr)
    return logger


def reopen_logs():
    """
    Ask handlers to reopen log stream if handler does support it.
    """
    logger = logging.getLogger()
    for handler in logger.handlers:
        if hasattr(handler, 'reopen'):
            try:
                handler.reopen()
            except EnvironmentError as e:
                logger.error(format_exc("failed to reopen log", e))
