# -*- coding: utf-8 -*-
"""
Some basic alemate models like instance, instance state etc

THIS FILE IS NOT SUPPORTED ANYMORE, AND PROBABLY OUTDATED, PLEASE DON'T USE IT
"""
import enum
import six

from sepelib.core.constants import MEGABYTE


class ContainerResourceLimits(object):
    """
    Instance limits description (like CPU, RAM, disk etc)
    """

    __slots__ = ['memory_guarantee', 'memory_limit',
                 'cpu_guarantee', 'cpu_limit', 'cpu_policy',
                 'io_limit', 'io_policy',
                 'net_guarantee', 'net_limit', 'net_priority', 'net_tos',
                 'disk_guarantee', 'disk_limit',
                 'dirty_limit', 'recharge_on_pgfault', 'ulimit']

    BYTE_LIMITS = ['memory_guarantee', 'memory_limit', 'io_limit', 'net_guarantee', 'net_limit', 'dirty_limit']
    CPU_LIMITS = ['cpu_guarantee', 'cpu_limit']
    DISK_LIMITS = ['disk_guarantee', 'disk_limit']
    NET_BYTE_LIMITS = ['net_guarantee', 'net_limit']

    def __init__(self, memory_guarantee=None, memory_limit=None, cpu_guarantee=None, cpu_limit=None, cpu_policy=None,
                 io_limit=None, io_policy=None, net_guarantee=None, net_limit=None, net_priority=None, net_tos=None,
                 disk_guarantee=None, disk_limit=None, dirty_limit=None, recharge_on_pgfault=None,
                 ulimit=None,
                 **_):
        """
        Instance limits description. For details see:
        https://wiki.yandex-team.ru/porto/limitsoverview
        https://wiki.yandex-team.ru/porto/propertiesanddata/
        https://wiki.yandex-team.ru/users/khlebnikov/yakernel/

        :param int memory_guarantee: Guaranteed memory in bytes for instance
        :param int memory_limit: Memory limit in bytes for instance
        :param float cpu_guarantee: Guaranteed CPU cores
        :param float cpu_limit: CPU cores
        :param cpu_policy: CPU policy
        :param int io_limit: Disk read/write speed limit (bytes/s)
        :param io_policy: Disk IO policy
        :param int net_guarantee: Guaranteed outgoing traffic (bytes/s)
        :param int net_limit: Outgoing traffic network limit (bytes/s)
        :param net_priority: Network packets priority
        :param net_tos: IP Type of Service
        :param disk_guarantee: Guaranteed disk space in Gb
        :param disk_limit: Disk space limit in Gb
        :param int dirty_limit: Dirty file cache limit (bytes)
        :param bool recharge_on_pgfault:
        :param unicode ulimit: ulimit specified as string for porto
        """
        self.memory_guarantee = memory_guarantee
        self.memory_limit = memory_limit
        self.cpu_guarantee = cpu_guarantee
        self.cpu_limit = cpu_limit
        self.cpu_policy = cpu_policy
        self.io_limit = io_limit
        self.io_policy = io_policy
        self.net_guarantee = net_guarantee
        self.net_limit = net_limit
        self.net_priority = net_priority
        self.net_tos = net_tos
        self.disk_guarantee = disk_guarantee
        self.disk_limit = disk_limit
        self.dirty_limit = dirty_limit
        self.recharge_on_pgfault = recharge_on_pgfault
        self.ulimit = ulimit

    @classmethod
    def from_dict(cls, params):
        return cls(**params)

    @classmethod
    def from_nanny_response(cls, params):
        # convert from Mb to bytes (from Mb/s to bytes/s)
        for limit in cls.BYTE_LIMITS:
            if limit in params:
                params[limit] *= MEGABYTE

        # convert from CPU milliseconds to cores
        for limit in cls.CPU_LIMITS:
            if limit in params:
                params[limit] /= 1000.0

        return ContainerResourceLimits.from_dict(params)

    @classmethod
    def from_allocation_order(cls, order):
        return ContainerResourceLimits(
            cpu_limit=order.cpu / 1000.0,
            memory_limit=order.memory * MEGABYTE,
            disk_limit=order.disk
        )

    @classmethod
    def from_gencfg_response(cls, params):
        params['cpu_limit'] = params.pop('cpu_cores_limit', None)
        params['cpu_guarantee'] = params.pop('cpu_cores_guarantee', None)
        return ContainerResourceLimits.from_dict(params)

    def __repr__(self):
        kwargs_str = []
        for attr in self.__slots__:
            kwargs_str.append('{}={!r}'.format(attr, getattr(self, attr, None)))
        return 'Instance({})'.format(kwargs_str)

    def update(self, other_limits):
        """
        Update limits with other limits
        """
        for attr in self.__slots__:
            limit = getattr(other_limits, attr, None)
            if limit is not None:
                setattr(self, attr, limit)

    def to_iss_dict(self):
        result = {}

        for attr in self.__slots__:
            # FIXME: add disk limits when supported by porto
            if attr not in self.DISK_LIMITS:
                value = getattr(self, attr, None)
                if value is not None:
                    result[attr] = value

        # porto expects CPU limits in core fraction format
        # e.g. 1.5c for 1.5 cores
        for attr in self.CPU_LIMITS:
            if attr in result:
                result[attr] = '{:f}c'.format(result[attr])

        # porto expects network limits in format "eth0: 100500; vlan761: 100500" or "default: 100500"
        for attr in self.NET_BYTE_LIMITS:
            if attr in result:
                result[attr] = 'default: {:d}'.format(result[attr])

        return result

    def to_dict(self):
        return {k: getattr(self, k, None) for k in self.__slots__}

    __str__ = __repr__


class Iss3HooksResourceLimits(object):

    __slots__ = ['iss_hook_stop', 'iss_hook_status', 'iss_hook_install', 'iss_hook_uninstall',
                 'iss_hook_validate', 'iss_hook_notify', 'iss_hook_reopenlogs']

    def __init__(self, iss_hook_stop=None, iss_hook_status=None, iss_hook_install=None, iss_hook_uninstall=None,
                 iss_hook_validate=None, iss_hook_notify=None, iss_hook_reopenlogs=None):
        self.iss_hook_stop = iss_hook_stop
        self.iss_hook_status = iss_hook_status
        self.iss_hook_install = iss_hook_install
        self.iss_hook_uninstall = iss_hook_uninstall
        self.iss_hook_validate = iss_hook_validate
        self.iss_hook_notify = iss_hook_notify
        self.iss_hook_reopenlogs = iss_hook_reopenlogs

    @classmethod
    def from_nanny_response(cls, params):
        return cls(**{f: ContainerResourceLimits.from_nanny_response(params.get(f, {})) for f in cls.__slots__})

    def to_iss_dict(self):
        """
        :rtype: dict
        """
        result = {}
        for hook in self.__slots__:
            hook_limits = getattr(self, hook)
            if hook_limits is not None:
                for limit_key, value in six.iteritems(hook_limits.to_iss_dict()):
                    result['{}.{}'.format(hook, limit_key)] = value
        return result


class MetaData(object):

    __slots__ = ['annotations']

    def __init__(self, annotations):
        """
        :type annotations: dict[unicode, unicode]
        """
        self.annotations = annotations

    @classmethod
    def from_nanny_response(cls, params):
        """
        :type params: dict
        :rtype: MetaData
        """
        annotations = params.get('annotations', [])
        return cls(annotations={l['key']: l['value'] for l in annotations})


class Iss3HooksTimeLimits(object):

    __slots__ = ['iss_hook_start', 'iss_hook_stop', 'iss_hook_status', 'iss_hook_install', 'iss_hook_uninstall',
                 'iss_hook_validate', 'iss_hook_notify', 'iss_hook_reopenlogs']

    def __init__(self, iss_hook_start=None, iss_hook_stop=None, iss_hook_status=None, iss_hook_install=None,
                 iss_hook_uninstall=None, iss_hook_validate=None, iss_hook_notify=None, iss_hook_reopenlogs=None):
        self.iss_hook_start = iss_hook_start
        self.iss_hook_stop = iss_hook_stop
        self.iss_hook_status = iss_hook_status
        self.iss_hook_install = iss_hook_install
        self.iss_hook_uninstall = iss_hook_uninstall
        self.iss_hook_validate = iss_hook_validate
        self.iss_hook_notify = iss_hook_notify
        self.iss_hook_reopenlogs = iss_hook_reopenlogs

    @classmethod
    def from_dict(cls, params):
        return cls(**{f: IssHookTimeLimits.from_dict(params.get(f, {})) for f in cls.__slots__})


class IssHookTimeLimits(object):

    __slots__ = [
        'min_restart_period',
        'max_execution_time',
        'restart_period_backoff',
        'max_restart_period',
        'restart_period_scale',
    ]

    def __init__(
        self,
        min_restart_period=None,
        max_execution_time=None,
        restart_period_backoff=None,
        max_restart_period=None,
        restart_period_scale=None,
    ):
        self.min_restart_period = min_restart_period
        self.max_execution_time = max_execution_time
        self.restart_period_backoff = restart_period_backoff
        self.max_restart_period = max_restart_period
        self.restart_period_scale = restart_period_scale

    @classmethod
    def from_dict(cls, params):
        return cls(**{f: params.get(f) for f in cls.__slots__})


class InvalidInstanceNetworkSettings(Exception):
    pass


class InstanceNetworkInterface(object):

    __slots__ = ['interface_name', 'address', 'hostname']

    def __init__(self, interface_name, address, hostname):
        """
        :type interface_name: unicode
        :type address: unicode
        :type hostname: unicode
        """
        self.interface_name = interface_name
        self.address = address
        self.hostname = hostname

    def __repr__(self):
        return 'InstanceNetworkInterface(interface_name={!r}, address={!r}, hostname={!r})'.format(
            self.interface_name, self.address, self.hostname
        )

    @classmethod
    def from_gencfg_response(cls, name, params):
        """
        :type name: unicode
        :type params: dict
        :rtype: InstanceNetworkInterface
        """
        address = params.get('ipv6addr')
        if not address:
            raise InvalidInstanceNetworkSettings('Invalid network settings in gencfg response: no ipv6addr given')
        hostname = params.get('hostname')
        if not hostname:
            raise InvalidInstanceNetworkSettings('Invalid network settings in gencfg response: no hostname given')

        return cls(interface_name=name,
                   address=address,
                   hostname=hostname)


class InstanceNetworkSettings(object):

    __slots__ = ['use_mtn', 'project_id', 'hostname', 'resolv_conf', 'interfaces', 'mtn_ready']

    def __init__(self, use_mtn, project_id, hostname, resolv_conf, interfaces, mtn_ready):
        """
        :type use_mtn: bool
        :type hostname: unicode
        :type project_id: unicode
        :type resolv_conf: unicode
        :type interfaces: list[InstanceNetworkInterface]
        :type mtn_ready: bool
        """
        self.mtn_ready = mtn_ready
        self.hostname = hostname
        self.use_mtn = use_mtn
        self.project_id = project_id
        self.resolv_conf = resolv_conf
        self.interfaces = interfaces

    @classmethod
    def from_gencfg_response(cls, d):
        """
        :type d: dict
        :rtype: InstanceNetworkSettings
        """
        mtn_ready = d.get('mtn_ready', False)
        project_id = d.get('hbf_project_id', '')
        resolv_conf = d.get('resolv.conf', '')
        dicts = d.get('interfaces', {})
        hostname = dicts.get('backbone', {}).get('hostname')
        interfaces = [InstanceNetworkInterface.from_gencfg_response(n, i) for n, i in six.iteritems(dicts)]
        return cls(
            use_mtn=False,
            project_id=project_id,
            hostname=hostname,
            resolv_conf=resolv_conf,
            interfaces=interfaces,
            mtn_ready=mtn_ready,
        )


class Instance(object):
    """
    Instance description
    """
    __slots__ = ['host', 'port', 'power', 'tags', 'short_hostname', 'shard', 'name', 'shard_info', 'virtual_shard',
                 'conf', 'gencfg_group', 'resource_limits', 'ipv4_address', 'ipv6_address', 'instance_cls',
                 'hooks_time_limits', 'annotated_ports', 'meta_data', 'hooks_resource_limits', 'location', 'dc',
                 'ports', 'network_settings']

    def __init__(self, host, port, power=0, tags=None, shard=None, name=None, shard_info=None, virtual_shard=None,
                 conf=None, gencfg_group=None, limits=None, ipv4_address=None, ipv6_address=None, instance_cls=None,
                 hooks_time_limits=None, annotated_ports=None, meta_data=None, hooks_resource_limits=None,
                 location=None, dc=None, ports=None, network_settings=None, **_):
        """
        :param host: хост инстанса
        :type host: str | unicode
        :param int port: порт инстанса
        :param power: мощность инстанса
        :type power: int | float
        :param list tags: список тегов
        :param str shard: название шарда
        :param BsconfigShard shard_info: информация про шард
        :param str virtual_shard: "виртуальный шард" -- строка по которой инстансу сопоставляется реальный
         шард при разворачивании шардмапа, например, 'primus-Rus-1-3-5-0000000000' => 'primus-Rus-1-3-5-1431890942'
        :param conf: имя конфигурации, в которую входит инстанс
        :type conf: str | unicode
        :param str gencfg_group: имя группы gencfg, в которую входит инстанс
        :param ContainerResourceLimits limits: ограничения на использование инстансом ресурсов сервера
        :param str ipv4_address: IPv4-адрес хоста
        :param str ipv6_address: IPv6-адрес хоста
        :param str instance_cls: название класса ISS инстанса
        :param hooks_time_limits: таймауты запуска status hook'а
        :type hooks_time_limits: Iss3HooksTimeLimits
        :param hooks_resource_limits: ограничения ресурсов и настройки porto для iss-хуков
        :type hooks_resource_limits: Iss3HooksResourceLimits
        :param annotated_ports: словарь именованных портов инстанса
        :type annotated_ports: dict[str | unicode, sepelib.yandex.nanny.AnnotatedPort]
        :type meta_data: MetaData
        :type ports: list[sepelib.yandex.nanny.AnnotatedPort]
        :type network_settings: InstanceNetworkSettings
        """
        self.host = host
        self.short_hostname = host.split('.', 1)[0]
        self.port = port
        self.power = power
        self.tags = tags if tags is not None else []
        self.shard = shard
        self.shard_info = shard_info
        self.virtual_shard = virtual_shard
        self.conf = conf
        self.gencfg_group = gencfg_group
        self.resource_limits = limits or ContainerResourceLimits()
        self.instance_cls = instance_cls
        self.hooks_time_limits = hooks_time_limits
        self.hooks_resource_limits = hooks_resource_limits
        self.ipv4_address = ipv4_address
        self.ipv6_address = ipv6_address
        self.name = name
        self.annotated_ports = annotated_ports or {}
        self.meta_data = meta_data or MetaData(annotations={})
        self.location = location
        self.dc = dc
        self.ports = ports or []
        self.network_settings = network_settings or InstanceNetworkSettings(use_mtn=False,
                                                                            project_id='',
                                                                            hostname='',
                                                                            resolv_conf='',
                                                                            interfaces=[],
                                                                            mtn_ready=False)
        if self.name is None:
            self.name = '{conf}@{host}:{port}'.format(conf=conf, host=host, port=port)

    def as_string(self):
        return "{}:{}@{}".format(self.short_hostname, self.port, self.conf)

    @classmethod
    def from_dict(cls, params):
        if 'limits' in params:
            params['limits'] = ContainerResourceLimits.from_nanny_response(params['limits'])
        return cls(**params)

    @classmethod
    def from_gencfg_response(cls, params):
        network = InstanceNetworkSettings.from_gencfg_response(params.get('hbf', {}))
        limits = params.get('porto_limits') or params.get('limits') or {}
        virtual_shard = params.get('shard_name')
        location_str = params.get('location', '').upper()
        location = Location.__members__.get(location_str)
        return cls(
            host=params['hostname'],
            port=params['port'],
            ipv4_address=params.get('ipv4addr'),
            ipv6_address=params.get('ipv6addr'),
            tags=params.get('tags', []),
            power=float(params.get('power', 1.0)),
            virtual_shard=virtual_shard if virtual_shard != "none" else None,
            limits=ContainerResourceLimits.from_gencfg_response(limits),
            location=location,
            network_settings=network,
        )

    def __repr__(self):
        kwargs_str = []
        for attr in self.__slots__:
            kwargs_str.append('{}={!r}'.format(attr, getattr(self, attr, None)))
        return 'Instance({})'.format(kwargs_str)

    @staticmethod
    def cmp(a, b):
        return (a > b) - (a < b)

    def __cmp__(self, other):
        return self.cmp(self.host, other.host) or self.cmp(self.port, other.port)

    def __hash__(self):
        return hash((self.host, self.port))

    def __eq__(self, other):
        return self.host == other.host and self.port == other.port

    __str__ = as_string


class Location(enum.Enum):
    MSK = 1
    SAS = 2
    MAN = 3
