"""
Auth (http://auth.qe.yandex-team.ru/) client.
"""
import socket
import six.moves.urllib.parse as urlparse

from requests.exceptions import RequestException

from sepelib.http.session import InstrumentedSession


class AuthQEError(Exception):
    pass


class AuthQEClient(object):
    _BASE_URL = 'https://auth.qe.yandex-team.ru/api/'
    _REQ_TIMEOUT = 10

    @classmethod
    def from_config(cls, d):
        return cls(base_url=d.get('base_url'),
                   req_timeout=d.get('req_timeout'),
                   verify_ssl=d.get('verify_ssl'))

    def __init__(self, base_url=_BASE_URL, req_timeout=_REQ_TIMEOUT, verify_ssl=None):
        self._base_url = base_url
        self._req_timeout = req_timeout or self._REQ_TIMEOUT
        self._verify_ssl = True if verify_ssl is None else verify_ssl
        self._session = InstrumentedSession('/clients/authqe')

    def get_roles(self, login):
        """
        Returns a list of user's roles.
        :rtype: list of strings
        """
        url = urlparse.urljoin(self._base_url, '/api/users/{0}'.format(login))
        try:
            resp = self._session.get(url, timeout=self._req_timeout, verify=self._verify_ssl)
        except socket.error as e:
            raise AuthQEError('socket error: {0}'.format(e.strerror))
        except RequestException as e:
            raise AuthQEError('connection error: {0}'.format(str(e)))

        content_type = resp.headers.get('content-type', 'text/xml').split(';')[0]
        if content_type != 'application/json':
            raise AuthQEError("received content type '{0}', "
                              "was waiting for JSON".format(content_type))

        if resp.status_code != 200:
            raise AuthQEError('bad http status: {0}'.format(resp.status_code))

        data = resp.json()
        return data['roles']

    # to be consistent with other services
    def start(self):
        pass

    def stop(self):
        self._session.close()
