"""
Simple center.yandex-team.ru API wrappers.
Docs: http://wiki.yandex-team.ru/intranet/center/api
"""
import json
import socket
import six

import requests
from requests.exceptions import RequestException, HTTPError


class CenterError(Exception):
    @property
    def url(self):
        return self.args[1]


class CenterClient(object):
    CENTER_URL = 'http://center-robot.yandex.net/api/v1'
    CENTER_TEST_URL = 'http://center.test.tools.yandex-team.ru/api/v1'
    _TOKEN = 'iss'  # very secret access token
    DEFAULT_REQ_TIMEOUT = 10  # seconds

    @classmethod
    def from_config(cls, d):
        return cls(center_url=d.get('center_url'),
                   token=d.get('token'),
                   req_timeout=d.get('req_timeout'),
                   trust_env=d.get('trust_env'))

    def __init__(self, center_url=None, token=None, req_timeout=None, trust_env=False):
        self._center_url = center_url or self.CENTER_URL
        self._token = token or self._TOKEN
        self._req_timeout = req_timeout or self.DEFAULT_REQ_TIMEOUT
        self._session = requests.Session()
        self._session.trust_env = trust_env

    def _http_get(self, url):
        try:
            response = self._session.get(url, timeout=self._req_timeout)
        except socket.error as e:
            raise CenterError("socket error: {0}".format(six.text_type(e)), self._center_url)
        except RequestException as e:
            raise CenterError('connection error: {0}'.format(six.text_type(e)), self._center_url)
        response.raise_for_status()
        try:
            return json.loads(response.content)
        except ValueError as e:
            raise CenterError("failed to parse response: "
                              "txt='{0}' err='{1}'".format(response.content, six.text_type(e)), self._center_url)

    def list_user_groups(self, login, fields=('id', 'name', 'url')):
        """
        List user groups accorinding to docs:
        http://wiki.yandex-team.ru/intranet/center/API#gruppyvkotoryxsostoitsotrudnik
        :param login: user login
        :param fields: list of returned fields (id|name|code|url|description)
        :return: list of user groups
        :rtype: list of dict
        """
        url = "{url}/user/{login}/groups.json?fields={fields}&token={token}".format(
            url=self._center_url, login=login, fields="|".join(fields), token=self._token)
        return self._http_get(url)

    def list_user_attributes(self, login, fields=('id', 'login')):
        """
        List user attributes according to docs:
        http://wiki.yandex-team.ru/intranet/center/API#odinochnyjjzapros
        :param login: user login
        :param fields: list of returned fields (id|login|first_name|last_name|work_email|work_phone|position)
        :return: dict with user attributes or None if user does not exists
        :rtype: dict
        """
        url = "{url}/user/{login}.json?fields={fields}&token={token}".format(
            url=self._center_url, login=login, fields="|".join(fields), token=self._token)
        try:
            return self._http_get(url)
        except HTTPError as e:
            if e.response is not None and e.response.status_code == 404:
                return None
            raise

    def list_group_attributes(self, group, fields=('id', 'name')):
        """
        List group attributes according to docs:
        http://wiki.yandex-team.ru/intranet/center/api#informacijaogruppe
        :param group: group identifier (url or id)
        :param fields: list of returned fields (id|parent|name|code|url)
        :return: dict with group attributes or None if group does not exists
        :rtype: dict
        """
        url = "{url}/groups/{group}.json?fields={fields}&token={token}".format(
            url=self._center_url, group=group, fields="|".join(fields), token=self._token)
        try:
            return self._http_get(url)
        except HTTPError as e:
            if e.response is not None and e.response.status_code == 404:
                return None
            raise

    def start(self):
        pass

    def stop(self):
        self._session.close()
