"""
Some utilities to interact with cluster master.
Upper level object is 'target'. Each target can be run on multiple hosts.
On each host each target can be represented as multiple tasks.

So target is succesfull if all it's tasks are sucessful on all hosts.
"""
from collections import namedtuple

import requests


REQ_TIMEOUT = 10
# CM_URL = 'http://ya.yandex.ru:5133/target_text'
CM_URL = 'http://localhost:5133/target_text'
SHARDMAP_TARGET = 'gen_shardmap'
SHARDS_TARGET = 'uploadsclusters_bsc_init_shard'


class HostTaskState(object):
    """
    Cluster master task state on specific host description.
    """
    Times = namedtuple('Times', ['started', 'finished', 'success', 'failed'])

    NOT_READY_STATES = frozenset(["idle", "pending", "ready", "running", "canceling", "canceled"])
    FAILED_STATES = frozenset(["skipped", "failed", "depfailed", "unknown"])
    SUCCESS_STATES = frozenset(["success"])

    @classmethod
    def from_line(cls, line):
        """
        Create CMTaskState class from :param line: received from cluster master
        """
        fields = line.split('\t')
        if len(fields) != 4:
            raise ValueError('bad task status line: {0}'.format(line))
        name, status, times_str, elapsed = fields
        times_parts = times_str.split()
        if len(times_parts) != 4:
            raise ValueError('bad times field: {0}'.format(times_str))
        times = cls.Times(*[int(i) for i in times_parts])
        return cls(name, status, elapsed, times)

    def __init__(self, name, status, elapsed, times):
        """
        :type name: str
        :type status: str
        :type elapsed: int
        :type times: Times
        :return:
        """
        self.name = name
        self.status = status
        self.elapsed = elapsed
        self.times = times

    def __str__(self):
        return "State(name={0} status={1} times={2})".format(
            self.name, self.status, self.times
        )

    @property
    def is_successful(self):
        return self.status in self.SUCCESS_STATES

    @property
    def last_success(self):
        return self.times.success or -1


class HostState(object):
    """
    Target specific tasks state on host.
    """
    def __init__(self, hostname, tasks):
        self.hostname = hostname
        self.tasks = tasks

    @property
    def is_successful(self):
        if not self.tasks:
            return False
        return all(task.is_successful for task in self.tasks)

    @property
    def last_success(self):
        if not self.tasks:
            return -1
        return max(i.last_success for i in self.tasks)


class Target(object):
    """
    Cluster master target descripting object.
    """
    def __init__(self, name, host_states):
        self.name = name
        self.host_states = host_states

    @property
    def is_successful(self):
        if not self.host_states:
            return False
        return all(i.is_successful for i in self.host_states)

    @property
    def last_success(self):
        if not self.host_states:
            return -1
        return max(i.last_success for i in self.host_states)


class ClusterMasterClient(object):
    @classmethod
    def from_config(cls, cfg):
        return cls(host=cfg['host'], port=cfg.get('port'))

    def __init__(self, host, port=None):
        self._host = host
        self._port = port or 5133
        self._base_url = "http://{0}:{1}/target_text".format(self._host, self._port)

    def _get_http_url(self, url, timeout=REQ_TIMEOUT):
        return requests.get(url, timeout=timeout).text

    def get_tasks_info(self, target, host, timeout=REQ_TIMEOUT):
        url = "{0}/{1}/{2}".format(self._base_url, target, host)
        text = self._get_http_url(url, timeout=timeout)
        tasks = []
        for line in text.splitlines():
            info = HostTaskState.from_line(line)
            tasks.append(info)
        return tasks

    def get_target_hosts(self, target, timeout=REQ_TIMEOUT):
        url = "{0}/{1}".format(self._base_url, target)
        text = self._get_http_url(url, timeout=timeout)
        # every line looks like this:
        # ya active success 1369133264 1369134238 1369134238 1360754094 0:16:14 0:16:14
        # have no idea what that means, we take only host information
        hosts = []
        for line in text.splitlines():
            host = line.split(None, 1)[0]
            hosts.append(host)
        return hosts

    def get_target_info(self, target_name, timeout=REQ_TIMEOUT):
        hosts = self.get_target_hosts(target_name, timeout)
        # now we have a list of hosts where target runs
        # let's gather info about every task on every host
        host_states = []
        for host in hosts:
            task_states = []
            for info in self.get_tasks_info(target_name, host, timeout):
                task_states.append(info)
            host_state = HostState(host, task_states)
            host_states.append(host_state)
        return Target(target_name, host_states)

    def get_shards_target_info(self):
        return self.get_target_info(SHARDS_TARGET)

production_cm_client = ClusterMasterClient('ya.yandex.ru')
