"""
Some wrappers around CMS (http://wiki.yandex-team.ru/JandeksPoisk/iss/cms) XMLRPC API.
"""
from collections import namedtuple
import json
import socket

import gevent
import requests
import six

from sepelib.http.session import InstrumentedSession


ResourceDescription = namedtuple('ResourceDescription', ['name', 'remote_path', 'localpath', 'chksum', 'arch'])


class FullInstanceName(namedtuple('FullInstanceName', ['host', 'port', 'conf'])):
    def as_string(self):
        return "{}:{}@{}".format(self.host, self.port, self.conf)


class CmsRequestError(Exception):
    pass


class CmsRequestTimeout(CmsRequestError):
    pass


class ICmsClient(object):
    """
    Interface to be used in dependency injection.
    """

    def list_configurations(self, name=None):
        """
        Calls 'listConf' to a list of all configurations if :param name: is None
        or a list with one description otherwise.
        Description is a dictionary with fields:
        {
            'name': 'BETA',
            'mtime': 1368812558
        }
        :rtype: list
        """
        raise NotImplementedError

    def get_configuration(self, name):
        """
        Simple helper.
        Calls 'listConf' with specified :param name: and return
        either configuration information or None
        """
        raise NotImplementedError

    def list_instance_tags(self, conf, host=None, instance_name=None):
        """
        Calls 'listAllSearchInstanceTags' and return a list of instance tags descriptions.
        Each description is a dictionary with fields:
        {
            'name': 'OPT_UltraBasePort=9587',
            'conf': 'BETA'
        }
        :param conf: configuration name
        :param host: limit results to specific host
        :param instance_name: limit results to specific instance
        :rtype: list
        """
        raise NotImplementedError

    def list_shard_tags(self, conf, shards=None, host=None):
        """
        Calls 'listAllShardsTags' and return a list of shard tags descriptions.
        Each description is a dictionary with fields:
        {
            'name': 'VideoQuickTier0',
            'conf': 'BETA',
            'shard': 'vidfidx-000-20130313-050000'
        }
        :param conf: configuration name
        :param shards: limit results to specific shards
        :param host: limit results to specific host
        :rtype: list
        """
        raise NotImplementedError

    def list_instances(self, conf, itag=None, stag=None, limit=None, order_by=None, add_fqdn=False):
        """
        Calls 'listSearchInstances' and return a list of instance descriptions.
        Each description is a dictionary with fields:
        {
            'shard': 'music_mic-173-1363528059',
            'host': 'ws40-493',
            'port': 8050,
            'name': 'ws40-493:8050@BETA'
        }
        :param conf: configuration name
        :param itag: limit results to instances with specific instance tag
        :param stag: limit results to instances on shard with specific shard tags
        :param limit: limit resulting list length
        :param order_by: order results by bs_searchinstance table column
        :param add_fqdn: add host FQDN in every instance description
        :rtype: list
        """
        raise NotImplementedError

    def list_instances_with_tags(self, conf):
        """
        Calls 'listSearchInstancesWithTags' and return a list of instance and tag pairs .
        Each description is a dictionary with fields:
        {
            'host': 'vsearch24-06',
            'port': 7300,
            'tag': 'MSK_DIVERSITY2_BASE@production_base-1463103409-auto'
        }
        :param conf: configuration name
        :rtype: list[dict]
        """
        raise NotImplementedError

    def list_hosts(self, conf, is_active=False):
        """
        Calls 'listHosts' to receive a list of hosts for :param conf:.
        Each element in resulting list is a dictionary with name field:
        {
            'name': 'imgs33-093'
        }
        :rtype: list
        """
        raise NotImplementedError

    def get_shards(self, shards):
        """
        Calls 'getShards' and get a list of :param shards: descriptions.
        At the time each description is a dictionary with fields:
        {
            'copy_size': '10288400548',
            'name': 'music_mic-078-1363528059',
            'shard_num': -1,
            'chksum': 'MD5:f0c5f61124d593852f4b91073d172d11',
            'priority': 0,
            'dep': '',
            'mtime': 1364333060,
            'resource_url': 'rbtorrent:ab37a3ea9b585c0e731577ec9b57fdfa626729ef'
        }
        :rtype: list
        """
        raise NotImplementedError

    def get_shard(self, shard_name):
        """
        Calls 'getShard' and returns shard description:
        {
            'copy_size': '10288400548',
            'name': 'music_mic-078-1363528059',
            'shard_num': -1,
            'chksum': 'MD5:f0c5f61124d593852f4b91073d172d11',
            'priority': 0,
            'dep': '',
            'mtime': 1364333060,
            'resource_url': 'rbtorrent:ab37a3ea9b585c0e731577ec9b57fdfa626729ef'
        }
        :rtype: dict
        """
        raise NotImplementedError

    def add_conf(self, conf_name):
        """
        Calls 'addConf'.
        If configuration exists - raises error.
        """
        raise NotImplementedError

    def add_host_conf(self, conf_name, is_active, host_or_hosts):
        """
        Calls 'addHostConf_bulk' setting :param conf_name: either active if
        :param is_active: == 1 or prepared if :param is_is_active: == 0 on host
        or list of hosts specified in :param host_or_hosts:
        """
        raise NotImplementedError

    def add_shard_to_hosts(self, conf_name, shard_to_hosts):
        """
        Calls 'addShardToHosts' adding shards to hosts.
        :type shard_to_hosts: list of pairs (shard_name, host_list), where `host_list`
                              is a list of host names
        """
        raise NotImplementedError

    def get_conf_dump(self, conf_name):
        """
        Calls 'getConfDump' returning configuration dump url to download.
        :param conf_name: configuration name
        :rtype: str
        """
        raise NotImplementedError

    def list_shard_on_hosts(self, conf_name):
        """
        Calls 'listShardOnHosts'.
        """
        raise NotImplementedError

    def add_instance(self, conf_name, shard_name, instances):
        """
        Calls 'addSearchInstance' adding instances to specified configuration.
        :type instances: list of :class:`FullInstanceName`
        """
        raise NotImplementedError

    def remove_conf(self, conf_name):
        """
        Calls 'removeConf' completely removing configuration from CMS.
        Raises error if configuration is active on some hosts.
        """
        raise NotImplementedError

    def configuration_exists(self, conf_name):
        """
        Calls 'confExists' to check if configuration :param conf_name: is registered in CMS.
        """
        raise NotImplementedError

    def add_resource(self, conf_name, resources):
        """
        Calls 'addResource' registering resource for specific configuration.
        :type conf_name: basestring
        :type resources: list of ResourceDescription
        """
        raise NotImplementedError

    def add_resource_to_instance(self, resource_name, conf_name, instance_names):
        """
        Calls 'addResourceToInstance' to add specified resource :param resource_name: to a set of instances
        identified as :param instance_names: in :param conf_name:.
        :type resource_name: basestring
        :type instance_names: list of FullInstanceName
        """
        raise NotImplementedError

    def set_instance_tag(self, tag, conf_name, instance_names):
        """
        Calls 'setSearchInstanceTag' to set :param tag: for instances in :param instance_descriptions: in
        configuration :param conf_name:
        :type instance_names: list of FullInstanceName
        """
        raise NotImplementedError

    def set_shard_tag(self, tag, conf_name, shard_names):
        """
        Calls 'setShardTag' to set :param tag: for shards in :param shard_names: in
        configuration :param conf_name:
        :type shard_names: list[str|unicode]
        """
        raise NotImplementedError

    def start(self):
        pass

    def get_metrics(self):
        raise NotImplementedError

    def stop(self):
        pass

    def list_resources(self, conf_name, return_raw=False):
        """
        Calls 'listResources' for configuration :param conf_name:
        If :param return_raw: is True - returns dictionary
        (useful in some cases to avoid extra transformations / copy operations).

        :type conf_name: str
        :type return_raw: bool
        :rtype: list of ResourceDescription

        """
        raise NotImplementedError

    def increment_conf_mtime(self, conf_name):
        """
        Calls 'incrementConfMtime' for configuration :param conf_name:, which
        increments configuration mtime by one.
        :type conf_name: str
        :return: updated configurations count (expecting one or zero)
        :rtype: int
        """
        raise NotImplementedError

    def list_resource_instances(self, resource_name, conf_name):
        """
            List configuration instances that contain specified resource

            :param resource_name: resource full name
            :type resource_name: str
            :param conf_name: configuration name
            :type conf_name: str
            :return: list of dicts with resource's instances
            :rtype: list of dict
        """
        raise NotImplementedError


class CmsClient(ICmsClient):
    CMS_RPC_URL = 'http://cmsearch.yandex.ru/json/bs'
    DEFAULT_REQ_TIMEOUT = 10  # seconds

    @classmethod
    def from_config(cls, d):
        return cls(cms_rpc_url=d.get('cms_rpc_url'),
                   req_timeout=d.get('req_timeout'))

    def __init__(self, cms_rpc_url=None, req_timeout=None):
        self._url = cms_rpc_url or self.CMS_RPC_URL
        self._timeout = req_timeout or self.DEFAULT_REQ_TIMEOUT
        self._session = InstrumentedSession('/clients/cms')

    def _jsonrpc_call(self, method, params):
        """
        Simple requests based XMRPC like JSON RPC.
        :param method: method name
        :param params: parameters for method being called
        :return: object
        """
        if not isinstance(params, tuple):
            params = params,
        data = json.dumps((method, params))
        with gevent.Timeout(self._timeout, CmsRequestTimeout):
            try:
                resp = self._session.post(self._url,
                                          data=data,
                                          headers={'Content-Type': 'application/json'},
                                          timeout=self._timeout)
            except socket.error as e:
                raise CmsRequestError("socket error: {0}".format(six.text_type(e)), self._url)
            except requests.Timeout as e:
                raise CmsRequestTimeout("request timeout: {0}".format(six.text_type(e)), self._url)
            resp.raise_for_status()
        try:
            answ, result = json.loads(resp.content)
        except ValueError as e:
            raise CmsRequestError("jsonrpc error: {0}".format(six.text_type(e)), self._url)
        if answ.lower() == 'ok':
            return result
        else:
            raise CmsRequestError('{0}: {1}'.format(answ, str(result)))

    def list_configurations(self, name=None):
        if name:
            filter_ = {'name': name}
        else:
            filter_ = {}
        return self._jsonrpc_call('listConf', filter_)

    def get_configuration(self, name):
        result = self.list_configurations(name)
        if result:
            return result[0]
        else:
            return None

    def list_instance_tags(self, conf, host=None, instance_name=None):
        filter_ = {
            'conf': conf
        }
        if host:
            filter_['host'] = host
        if instance_name:
            filter_['instanceName'] = instance_name
        return self._jsonrpc_call('listAllSearchInstanceTags', filter_)

    def list_shard_tags(self, conf, shards=None, host=None):
        filter_ = {
            'conf': conf,
            'shards': shards or []
        }
        if host:
            filter_['host'] = host
        return self._jsonrpc_call('listAllShardsTags', filter_)

    def list_instances(self, conf, itag=None, stag=None, limit=None, order_by=None, add_fqdn=False):
        filter_ = {
            'conf': conf
        }
        if itag:
            filter_['instanceTagName'] = itag
        if stag:
            filter_['shardTagName'] = stag
        if limit:
            filter_['limit'] = limit
        if add_fqdn:
            filter_['addFqdn'] = True
        if order_by is None:  # don't pass None via XML RPC
            order_by = ''
        return self._jsonrpc_call('listSearchInstances', (filter_, order_by))

    def list_instances_with_tags(self, conf):
        return self._jsonrpc_call('listSearchInstancesWithTags', conf)

    def list_hosts(self, conf, is_active=False):
        # cms api quirk
        if is_active:
            params = (conf, 1)
        else:
            params = (conf,)
        return self._jsonrpc_call('listHosts', params)

    def get_shards(self, shards):
        return self._jsonrpc_call('getShards', shards)

    def get_shard(self, shard_name):
        return self._jsonrpc_call('getShard', shard_name)

    def add_conf(self, conf_name):
        return self._jsonrpc_call('addConf', conf_name)

    def add_host_conf(self, conf_name, is_active, host_or_hosts):
        if not isinstance(host_or_hosts, list):
            host_or_hosts = [host_or_hosts]
        return self._jsonrpc_call('addHostConf_bulk', (host_or_hosts, conf_name, is_active))

    def add_shard_to_hosts(self, conf_name, shard_to_hosts):
        prepared_data = [(shard_name, ' '.join(host_list)) for shard_name, host_list in shard_to_hosts]
        return self._jsonrpc_call('addShardToHosts', (conf_name, prepared_data))

    def get_conf_dump(self, conf_name):
        return self._jsonrpc_call('getConfDump', (conf_name,))

    def list_shard_on_hosts(self, conf_name):
        return self._jsonrpc_call('listShardOnHosts', {'conf': conf_name})

    def add_instance(self, conf_name, shard_name, instances):
        instances = [(shard_name, i.host, i.port) for i in instances]
        return self._jsonrpc_call('addSearchInstance', (conf_name, instances))

    def remove_conf(self, conf_name):
        return self._jsonrpc_call('removeConf', conf_name)

    def configuration_exists(self, conf_name):
        return self._jsonrpc_call('confExists', conf_name)

    def add_resource(self, conf_name, resources):
        return self._jsonrpc_call('addResource', (conf_name, resources))

    def add_resource_to_instance(self, resource_name, conf_name, instance_names):
        instance_names = [i.as_string() for i in instance_names]
        return self._jsonrpc_call('addResourceToInstance', ('{}@{}'.format(resource_name, conf_name), instance_names))

    def set_instance_tag(self, tag, conf_name, instance_names):
        # cms awaits tag in 'full' representation
        fulltag = '{}@{}'.format(tag, conf_name)
        instance_names = [i.as_string() for i in instance_names]
        return self._jsonrpc_call('setSearchInstanceTag', (fulltag, instance_names))

    def set_shard_tag(self, tag, conf_name, shard_names):
        # cms awaits tag in 'full' representation
        fulltag = '{}@{}'.format(tag, conf_name)
        return self._jsonrpc_call('setShardTag', (fulltag, shard_names))

    def start(self):
        pass

    def get_metrics(self):
        return self._session.get_metrics()

    def stop(self):
        self._session.close()

    def list_resources(self, conf_name, return_raw=False):
        raw_result = self._jsonrpc_call('listResources', {'conf': conf_name})
        if return_raw:
            return raw_result
        else:
            return [ResourceDescription(r['name'],
                                        r['remote_path'],
                                        r['local_path'],
                                        r['chksum'],
                                        r['arch'])
                    for r in raw_result]

    def increment_conf_mtime(self, conf_name):
        return self._jsonrpc_call('incrementConfMtime', (conf_name,))

    def list_resource_instances(self, resource_name, conf_name):
        """
            List configuration instances that contain specified resource

            :param resource_name: resource's name
            :param conf_name: CMS configuration name
            :return: list of instances names
            :rtype: list of str
        """
        instances = self._jsonrpc_call(
            'listSearchInstancesByResource',
            '{}@{}'.format(resource_name, conf_name)
        )
        return [instance.get('instance') for instance in instances if instance.get('instance')]
