# -*- coding: utf-8 -*-
"""
DNS API v2 client https://wiki.yandex-team.ru/dynamic-dns/dns-api-v2/
"""
import logging
from six.moves import http_client as httplib
import json
import time
import uuid
import six
from six.moves.urllib.parse import urljoin, urlparse
from requests.exceptions import Timeout, ConnectionError

from requests.exceptions import RequestException
import object_validator
from object_validator import DictScheme, String, List, Integer

from sepelib.core import constants
from sepelib.http.session import InstrumentedSession
from . import ApiRequestException


class DnsApiError(ApiRequestException):
    pass


class DnsApiMalformedResponse(DnsApiError):
    pass


class DnsApiNotFound(DnsApiError):
    pass


class DnsApiJobNotReady(DnsApiError):
    pass


class DnsApiTemporaryError(DnsApiError):
    pass


class DnsApiJobFailed(DnsApiError):
    def __init__(self, job_id, expression, error_description, **kwargs):
        self.job_id = job_id
        self.expression = expression or ""
        self.error_description = error_description
        super(DnsApiJobFailed, self).__init__(**kwargs)

    def __str__(self):
        return "DNS-API job {} ({}) has failed: {}".format(self.job_id, self.expression, self.error_description)


DEFAULT_BASE_URL = "https://dns-api.yandex.net/v2.3/"


class _DnsApiNetworkClient(object):
    def __init__(self, url_prefix, cert, key, token, timeout=constants.NETWORK_TIMEOUT, validate_only=True, log=None):
        self.session = InstrumentedSession("dns-api-client")
        self.cert = cert
        self.key = key
        self.token = token
        self.timeout = timeout
        self.validate_only = validate_only
        self.log = log or logging.getLogger(__name__ + ".network")
        self.url_prefix = url_prefix
        # Can be useful to debug dns-api errors
        self.last_request = None
        self.last_response = None

    def mk_url(self, url):
        return urljoin(self.url_prefix, url)

    def api_request(self, method, resource, comma_separated_params=None, key_value_params=None, payload=None, scheme=None, **kwargs):
        resource_url = self.mk_url(resource)

        if resource == "primitives" and self.validate_only:
            comma_separated_params = list(comma_separated_params or []) + ["validating"]

        if comma_separated_params:
            resource_url += "?" + ",".join(comma_separated_params)
        response = self._request(method, resource_url, params=key_value_params, payload=payload, scheme=scheme, **kwargs)
        return response

    def _json_request(self, method, url, params=None, data=None, scheme=None, **kwargs):
        request_id = uuid.uuid4().hex

        try:
            self.log.debug("Request request_id={}: {method} {url}, params={params}, validate_only={validate_only}".format(
                request_id,
                method=method,
                url=url,
                params=repr(params),
                validate_only=self.validate_only,
            ))
            if data is not None:
                self.log.debug("Request request_id={} data: {}".format(
                    request_id,
                    repr(data),
                ))

            response = self.session.request(method, url,
                                            params=params,
                                            data=data,
                                            **kwargs)

            self.last_request = response.request
            self.last_response = response

            self.log.debug("Response request_id={}: {} {}".format(
                request_id,
                response.status_code,
                response.reason,
            ))
            self.log.debug("Response request_id={} data: {}".format(
                request_id,
                repr(response.content),
            ))
        except (ConnectionError, Timeout) as e:
            self.last_request = None
            self.last_response = None
            raise DnsApiTemporaryError(six.text_type(e), response=e.response, request=e.request)
        except RequestException as e:
            self.last_request = None
            self.last_response = None
            raise DnsApiError(six.text_type(e),
                              response=e.response, request=e.request)

        # All possible client error codes are:
        # httplib.UNAUTHORIZED: OAuth token is invalid.
        # httplib.NOT_FOUND: item not found. It is a permanent error.
        # httplib.BAD_REQUEST, httplib.CONFLICT, httplib.NOT_ACCEPTABLE: invalid request. It is a client error.

        if response.status_code == httplib.NOT_FOUND:
            raise DnsApiNotFound(u"Method: {}, URL: {}, Params: {}".format(method, url, repr(params)),
                                 response=response)

        if response.status_code in [httplib.BAD_REQUEST, httplib.NOT_ACCEPTABLE, httplib.CONFLICT]:
            # Something went wrong, probably DNS-record already exists.
            # You need to check it with slayer@
            raise DnsApiError(u"Server returned {} {}".format(response.status_code, response.reason),
                              response=response)

        if response.status_code not in [httplib.OK, httplib.ACCEPTED]:
            raise DnsApiTemporaryError(u"Server returned {} {}".format(response.status_code, response.reason),
                                       response=response)

        try:
            parsed_response = response.json()
        except ValueError as e:
            self.log.debug("Malformed response: {}".format(repr(response.content)))
            raise DnsApiMalformedResponse(six.text_type(e),
                                          response=response)

        if scheme is not None:
            try:
                object_validator.validate("response", parsed_response, scheme)
            except object_validator.ValidationError as e:
                raise DnsApiMalformedResponse(six.text_type(e),
                                              response=response)
        return parsed_response

    def _request(self, method, url, params=None, payload=None, scheme=None):
        headers = {
            "Accept": "application/json",
            "X-Auth-Token": self.token,
            "Content-Type": "application/json",
        }
        if self.validate_only:
            headers["X-Api-Request-Mode"] = "ValidateOnly"

        response = self._json_request(method, url,
                                      params=params,
                                      data=payload,
                                      cert=(self.cert, self.key),
                                      verify=False,
                                      headers=headers,
                                      scheme=scheme,
                                      )
        return response


class ZoneRecordsCursor(object):
    def __init__(self, zone_id, zone_records_response, network_client):
        self.network_client = network_client
        self.zone_id = zone_id
        self.zone_url = self.network_client.mk_url("zones/{}/records".format(zone_id))
        self.zone_records_response = zone_records_response
        self.binary_blob = zone_records_response.get("blob", None)

    def get_next_page_url(self):
        links = self.zone_records_response["links"]
        next_page_url = [l["href"] for l in links if l["rel"] == "next"]
        if next_page_url:
            # http://st.yandex-team.ru/DNS-188
            # DNS API returns broken URLs with rel=next, so we construct
            # our own URL but with parameters taken from rel=next url.
            next_page_query = urlparse(next_page_url[0]).query
            next_page_url = self.zone_url + "?" + next_page_query
            return next_page_url
        else:
            # If rel=next URL is not present, we've reached the last page.
            return None

    def __iter__(self):
        while True:
            try:
                records = self.zone_records_response["recordsList"]["records"]
            except KeyError:
                raise
            for record in records:  # py3 "yield from", I miss you :'-(
                yield record

            next_page_url = self.get_next_page_url()
            if next_page_url:
                response = self.network_client.api_request("GET", next_page_url)
                self.zone_records_response = response["zones"][0]
            else:
                break

    def __repr__(self):
        return "<{}: zone_id={} zone_url={}>".format(self.__class__.__name__,
                                                     self.zone_id,
                                                     self.zone_url,
                                                     )


class DnsApiJobStatusTracker(object):
    def __init__(self, job_id, network_client):
        self.job_id = job_id
        self.network_client = network_client
        self.status = None
        self.expression = None
        self.error_code = None
        self.error_message = None
        self.error_details = None
        self._response = None

    def _get_status(self):
        response_scheme = DictScheme({
            "jobId": String(),
            "status": String(),
            "response": DictScheme({
                "expression": String(optional=True),
            }, optional=True, ignore_unknown=True),
            "error": DictScheme({
                "code": Integer(optional=True),
                "details": String(optional=True),
                "message": String(optional=True),
            }, optional=True, ignore_unknown=True),
        }, ignore_unknown=True)
        resource = "status/{}".format(self.job_id)
        response = self.network_client.api_request("GET", resource, scheme=response_scheme)
        self._response = response
        return response

    def update_job_status(self):
        response = self._get_status()
        self.status = response["status"]

        self.expression = response.get("response", {}).get("expression")
        error_data = response.get("error")
        if error_data is not None:
            self.error_code = error_data.get("code")
            self.error_message = error_data.get("message")
            self.error_details = error_data.get("details")

    def get_current_status(self):
        self.update_job_status()
        return self.status

    def wait(self, interval=1.0, attempts=None, timeout=None):
        used_attempts = 0
        timeout_time = None if timeout is None else time.time() + timeout

        while True:
            if attempts is not None and used_attempts == attempts:
                raise DnsApiJobNotReady("Job {} isn't ready yet".format(self.job_id))

            used_attempts += 1
            status = self.get_current_status()
            if status == "ERROR":
                if self.error_code or self.error_message or self.error_details:
                    error_description = "{} {}. {}.".format(self.error_code, self.error_message, self.error_details)
                else:
                    error_description = "No error message provided by DNS-API."
                raise DnsApiJobFailed(self.job_id, self.expression, error_description,
                                      response=self._response)
            if status == "COMPLETED":
                return

            if timeout_time is not None and time.time() + interval >= timeout_time:
                raise DnsApiJobNotReady("Job {} isn't ready yet".format(self.job_id))

            time.sleep(interval)

    def __repr__(self):
        return "<{}: job_id={} status={}>".format(self.__class__.__name__,
                                                  self.job_id,
                                                  self.status,
                                                  )


class DnsApiClient(object):
    SCHEMAS = {
        "raw_add_records": DictScheme({"jobId": String()}, ignore_unknown=True),
        "job_status": DictScheme({"status": String()}, ignore_unknown=True),
        "zone_records": DictScheme({
            "zones": List(DictScheme({
                "uuid": String(),
                "recordsList": DictScheme({
                    "records": List(DictScheme({
                        "left-side": String(),
                        "right-side": String(),
                        "ttl": String(),
                        "type": String(),
                    }, ignore_unknown=True)),
                }, ignore_unknown=True),
            }, ignore_unknown=True)),
        }, ignore_unknown=True),
        "record_info": DictScheme({
            "records": List(DictScheme({
                "left-side": String(),
                "right-side": String(),
                "type": String(),
                "metadata": DictScheme({"id": String()}, ignore_unknown=True),
            }, ignore_unknown=True)),
        }, ignore_unknown=True),
    }

    """DNS API v2 client."""
    def __init__(self, cert, key, login, token,
                 validate_only=True, base_url=DEFAULT_BASE_URL, timeout=constants.NETWORK_TIMEOUT):
        """
        :param cert: path to a client certificate (public part)
        :param key: path to a private key
        :param login: username under which all operations will be performed
        :param token: OAuth token to access DNS API
        :param timeout: Network timeout, either float or a pair of floats (connect_timeout, read_timeout)
        :param validate_only: if True, no changes will be made in DNS (dry-run)
        :param base_url: Base URL of DNS API
        :return:
        """
        self.cert = cert
        self.key = key
        self.login = login
        self.token = token
        self.timeout = timeout
        self.url_prefix = urljoin(base_url, login + "/")

        self.network_client = _DnsApiNetworkClient(self.url_prefix, self.cert, self.key, self.token,
                                                   timeout=self.timeout, validate_only=validate_only)

    # Raw methods, i.e. without any response processing

    def raw_add_records(self, records):
        """
        :param records: List of tuples (left side, MX/PTR/A/AAAA/whatever, right side, ttl)
        """
        records_data = []
        for name, record_type, data, ttl in records:
            ttl = int(ttl)
            record_data = {"name": name, "type": record_type, "data": data, "ttl": ttl}
            records_data.append(record_data)

        request_data = {"records": records_data}
        request_json = json.dumps(request_data)

        response = self.network_client.api_request("POST", "records", payload=request_json, scheme=self.SCHEMAS["raw_add_records"])
        return response

    def raw_job_status(self, job_id):
        resource = "status/{}".format(job_id)
        response = self.network_client.api_request("GET", resource, scheme=self.SCHEMAS["job_status"])
        return response

    def raw_zone_records(self, zone_id,
                         show_records=True, show_blob=False, show_metadata=False,
                         limit=None, offset=None,
                         ):
        """
        :param zone_id: Zone id.
        :param show_records: Return associated DNS records with the given zone.
        :param show_blob: Return binary blob (???).
        :param show_metadata: Return metadata for each record.
        :return: Returns DNS records for the given zone.
        """
        resource = "zones/{}/records".format(zone_id)

        show_params = []
        if show_records:
            show_params.append("showRecords")
        if show_blob:
            show_params.append("showBlob")
        if show_metadata:
            show_params.append("showMetadata")

        params = {}
        if limit is not None:
            params["limit"] = limit
        if offset is not None:
            params["offset"] = offset

        response = self.network_client.api_request("GET", resource,
                                                   comma_separated_params=show_params, key_value_params=params,
                                                   scheme=self.SCHEMAS["zone_records"])
        return response

    def apply_primitives(self, primitives, show_operations=False):
        payload = json.dumps({"primitives": primitives})
        return self.network_client.api_request("PUT", "primitives", payload=payload,
                                               comma_separated_params=["showOperations"] if show_operations else None)

    # More or less user-friendly methods

    def tokens(self):
        """
        :return: Information about used token.
        """
        response = self.network_client.api_request("GET", "tokens")
        return response

    def zone_info(self, name):
        """
        :param name: CIDR, ARPA or direct address.
        :return: Zone configuration.
        :raise: DnsApiNotFound.
        """

        try:
            zones = self.zones_info([name])["zones"]
        except DnsApiNotFound:
            raise DnsApiNotFound("Zone '{}' doesn't exist.".format(name))

        if len(zones) != 1:
            raise DnsApiError("Got an invalid response from DNS API.")

        return zones[0]

    def zones_info(self, names):
        """
        :param names: List of CIDRs, ARPAs or direct addresses.
        :return: Zone configuration.
        """
        response = self.network_client.api_request("GET", "zones", comma_separated_params=names)
        return response

    def zones_by_addrs(self, names):
        """
        :param names: List of domain names or IP addresses.
        :return: Zones associated with the given names or addresses.
        """
        response = self.network_client.api_request("GET", "zones/map", comma_separated_params=names)
        return response

    def zone_records(self, zone_id,
                     show_metadata=False, show_blob=False,
                     limit=None, offset=None,
                     ):
        """
        :param zone_id: Zone id.
        :param show_metadata: Return metadata for each record.
        :param show_blob: Return binary blob (???).
        :return: Returns DNS records wrapped into an iterator.
        """
        response = self.raw_zone_records(zone_id,
                                         show_metadata=show_metadata, show_blob=show_blob, show_records=True,
                                         limit=limit, offset=offset)

        if not response["zones"]:
            raise DnsApiMalformedResponse("empty \"zones\" key")
        zone = response["zones"][0]
        return ZoneRecordsCursor(zone["uuid"],
                                 zone,
                                 network_client=self.network_client,
                                 )

    def record_info(self, name, record_type=None):
        """
        :param name: IP or domain name (the so-called DNS record's "left side")
        :param record_type: MX, PTR, A, AAAA, whatever
        :return: Returns the so-called DNS record's "right side"
        """
        params = {"name": name}
        if record_type is not None:
            params["type"] = record_type

        response = self.network_client.api_request("GET", "records", key_value_params=params,
                                                   scheme=self.SCHEMAS["record_info"])
        return response

    def add_record(self, name, record_type, data, ttl):
        """
        :param name: Left side
        :param record_type: MX, PTR, A, AAA, whatever
        :param data: Right side
        :param ttl: TTL
        """
        return self.add_records([(name, record_type, data, ttl)])

    def add_records(self, records):
        response = self.raw_add_records(records)
        job_id = response["jobId"]
        return self.job_status(job_id)

    def delete_record(self, record_id):
        return self.delete_records([record_id])

    def delete_records(self, record_ids):
        response = self.network_client.api_request("DELETE", "records", comma_separated_params=record_ids)
        job_id = response["jobId"]
        return self.job_status(job_id)

    def job_status(self, job_id):
        return DnsApiJobStatusTracker(job_id, network_client=self.network_client)
