# coding: utf-8
import re
from six.moves.urllib.parse import urljoin

import inject
import requests
from requests.adapters import HTTPAdapter

from sepelib.http.session import InstrumentedSession


class IDockerResolverClient(object):
    @classmethod
    def instance(cls):
        """
        :rtype: IDockerResolverClient
        """
        return inject.instance(cls)

    def get_image_manifest(self, registry_url, image_name):
        """
        :type registry_url: str | unicode
        :type image_name: str | unicode
        :rtype: dict
        """
        raise NotImplementedError

    def get_layers_urls(self, registry_url, image_name):
        """
        :type registry_url: str | unicode
        :type image_name: str | unicode
        :rtype: list[str]
        """
        raise NotImplementedError


class DockerResolverClient(IDockerResolverClient):
    _DEFAULT_BASE_URL = 'https://dockinfo.yandex-team.ru/api/docker/'
    _DEFAULT_REQ_TIMEOUT = 30
    _DEFAULT_ATTEMPTS = 3
    ROBOT_USER_LOGIN = 'robot-qloud-client'
    image_name_pattern = re.compile(r'^(?P<image>[^:@]+)(:(?P<tag>[^:@]+)|@(?P<digest>sha256:[^:@]+))?$')

    def __init__(self, url=None, req_timeout=None, attempts=None, ssl_verify=None):
        self.api_url = url or self._DEFAULT_BASE_URL
        self.req_timeout = req_timeout or self._DEFAULT_REQ_TIMEOUT
        self.attempts = int(attempts if attempts is not None else self._DEFAULT_ATTEMPTS)
        if self.attempts < 1:
            raise ValueError('attempts must be >= 1, got {}'.format(self.attempts))
        self._session = InstrumentedSession('/clients/docker-resolver')
        self._session.mount(self.api_url, self._make_http_adapter())
        self._session.verify = ssl_verify

    def _make_http_adapter(self):
        max_conn = max(1, self.attempts / 2)
        return HTTPAdapter(pool_connections=max_conn, pool_maxsize=max_conn, max_retries=self.attempts)

    @classmethod
    def from_config(cls, d):
        return cls(**d)

    def get_image_manifest(self, registry_url, image_name):
        image_name, tag, digest = self._parse_image_name(image_name)
        if not tag:
            raise ValueError('Only Image:Tag notation supported')
        url = 'resolve?registryUrl={}/{}&tag={}'.format(registry_url, image_name, tag)
        response = self._session.request('get', urljoin(self.api_url, url),
                                         timeout=self.req_timeout)
        try:
            response.raise_for_status()
        except requests.HTTPError as e:
            if e.response.status_code == 404:
                raise ValueError('Image {} not found in registry {}'.format(image_name, registry_url))
            elif e.response.status_code == 403:
                raise ValueError('User "{}" does not have access to image {}'.format(
                    self.ROBOT_USER_LOGIN,
                    image_name
                ))
            else:
                raise
        return response.json()

    def get_layers_urls(self, registry_url, image_name):
        image_info = self.get_image_manifest(registry_url, image_name)
        # layers stored in reverse order
        return image_info['layers'][::-1]

    def _parse_image_name(self, full_image_name):
        """
        :type full_image_name: str | unicode
        :rtype: tuple(str, str | None, str | None)
        """
        match = self.image_name_pattern.match(full_image_name)
        if not match:
            raise ValueError('Invalid image name {}'.format(full_image_name))
        parsed = self.image_name_pattern.match(full_image_name).groupdict()
        return parsed['image'], parsed['tag'], parsed['digest']
