"""
Wrappers around hivemind API (http://wiki.yandex-team.ru/jandekspoisk/iss/hivemind#interfejjsvzaimodejjstvijaapiv.1)
Experimenting with stacked API:
  * client groups() method returns _Groups object
  * groups exports get_group() method, which returns _Group object
  * invoking .list_hosts on _Group object return list of hosts
"""
import requests


class _Group(object):
    def __init__(self, hm_client, group):
        self._hm_client = hm_client
        self._group = group

    def list_hosts(self):
        url = "{0}/groups/{1}/hosts?fmt=txt".format(self._hm_client.base_url, self._group)
        body = self._hm_client._http_get(url)
        return body.split()


class _Groups(object):
    def __init__(self, hm_client):
        self._hm_client = hm_client

    def get_group(self, group):
        return _Group(self._hm_client, group)


class HivemindClient(object):
    @staticmethod
    def _http_get(url, timeout=100):
        try:
            resp = requests.get(url, timeout=timeout)
        except requests.exceptions.Timeout:
            raise Exception("request timed out", url)
        resp.raise_for_status()
        return resp.content

    @classmethod
    def from_config(cls, cfg):
        """
        Create new client instance with parameters from configuration dict.
        """
        return cls(host=cfg['host'], port=cfg.get('port'))

    def __init__(self, host, port=None):
        self._host = host
        self._port = port or 80
        self._base_url = 'http://{0}:{1}/v1'.format(self._host, self._port)

    @property
    def base_url(self):
        return self._base_url

    def groups(self):
        return _Groups(self)
