# -*- coding: utf-8 -*-
"""
Http-GridFS (https://git.qe-infra.yandex-team.ru/projects/NANNY/repos/http-gridfs/) client.
"""
from six.moves import http_client as httplib
from object_validator import DictScheme, String

from sepelib.http.request import json_request
from sepelib.http.session import InstrumentedSession
from sepelib.yandex import ApiRequestException


class HttpGridfsApiRequestException(ApiRequestException):
    """Common-case exception for http-gridfs service request"""


class IHttpGridfsClient(object):
    """
    Interface to be used in dependency injection.
    """
    def put(self, content, content_type=None):
        pass


class HttpGridfsClient(IHttpGridfsClient):
    _DEFAULT_BASE_URL = 'http://http-gridfs.yandex-team.ru/v1'
    _DEFAULT_REQ_TIMEOUT = 30
    _DEFAULT_ATTEMPTS = 3

    PUT_RESOURCE_RESPONSE_SCHEME = DictScheme({
        "id": String(),
    }, ignore_unknown=True)

    @classmethod
    def from_config(cls, d):
        return cls(**d)

    def __init__(self, url=None, req_timeout=None, attempts=None):
        self._base_url = url or self._DEFAULT_BASE_URL
        self._req_timeout = req_timeout or self._DEFAULT_REQ_TIMEOUT
        self._attempts = attempts or self._DEFAULT_ATTEMPTS
        self._session = InstrumentedSession('/clients/httpgridfs')

    def put(self, content, content_type=None):
        """
        Put file to GridFS

        :type content: str
        :type content_type: str
        :return: created resource url
        :rtype: str
        """
        url = '{base_url}/resources/'.format(base_url=self._base_url)
        response = json_request(
            'post', url, scheme=self.PUT_RESOURCE_RESPONSE_SCHEME, ok_statuses=[httplib.OK, httplib.CREATED],
            timeout=self._req_timeout, exception=HttpGridfsApiRequestException, session=self._session,
            data=content, headers={'Content-Type': content_type}
        )
        return '{base_url}/resources/{id}/'.format(base_url=self._base_url, id=response['id'])
