# -*- coding: utf-8 -*-
"""
Instance Tune System client https://wiki.yandex-team.ru/jandekspoisk/sepe/nanny/its
"""
import os
import json
import six
import time
from six.moves import http_client as httplib
from collections import namedtuple

import requests
from object_validator import DictScheme, String, List, Dict
from werkzeug.http import parse_dict_header

from sepelib.http.request import json_request
from sepelib.http.session import InstrumentedSession
from sepelib.yandex import ApiRequestException
from sepelib.http import request


HeadInstance = namedtuple('HeadInstance', ['host', 'port'])


ItsControls = namedtuple('ItsControls', ['controls', 'cache_time', 'etag', 'version'])


ItsControl = namedtuple('ItsControl', ['control', 'etag'])


class ItsApiRequestException(ApiRequestException):
    """Common-case exception for ITS service request"""
    pass


class ItsConcurrentModificationException(ApiRequestException):
    pass


class IItsClient(object):
    """
    Interface to be used in dependency injection.
    """

    def resolve_instance_filter(self, instance_filter):
        """
        Resolve blinov filter which contains itags only using HEAD

        :type str instance_filter:
        :return: list of instance names
        :rtype: list
        """
        raise NotImplementedError

    def get_controls(self, itags, etag=None, expect_ok=False):
        """
        Get controls values for given itags list

        :type itags: list of str
        :type etag: str
        :type expect_ok: bool
        :param expect_ok: ask ITS to return values even if
        given etag matches last version of controls
        :rtype: ItsControls
        """
        raise NotImplementedError

    def get_control_value(self, control_path):
        """
        Gets ITS control value

        :type control_path: basestring
        :rtype: ItsControls|None
        """
        raise NotImplementedError

    def set_control_value(self, control_path, value, etag=None):
        """
        Sets ITS control value

        :type control_path: basestring
        :type value: basestring
        :type etag: basestring|None
        """
        raise NotImplementedError

    def delete_control_value(self, control_path, etag=None):
        """
        Deletes ITS control value

        :type control_path: basestring
        :type etag: basestring|None
        """
        raise NotImplementedError


class ItsClient(IItsClient):
    _DEFAULT_BASE_URL = 'http://its.yandex-team.ru/v1'
    _DEFAULT_REQ_TIMEOUT = 30
    _DEFAULT_ATTEMPTS = 3

    RESOLVE_INSTANCE_FILTER_SCHEME = DictScheme({
        "result": List(String())
    })

    ITS_CONTROLS_SCHEME = Dict(key_scheme=String())

    @classmethod
    def from_config(cls, d):
        return cls(**d)

    def __init__(self, url=None, req_timeout=None, attempts=None, token=None):
        self._base_url = url or self._DEFAULT_BASE_URL
        self._req_timeout = req_timeout or self._DEFAULT_REQ_TIMEOUT
        self._attempts = attempts or self._DEFAULT_ATTEMPTS
        self._session = InstrumentedSession('/clients/its')
        self._token = token

    def resolve_instance_filter(self, instance_filter):
        """
        Resolve blinov filter which contains itags only using HEAD

        :type str instance_filter:
        :return: list of instance names
        :rtype: list
        """
        url = '{base_url}/resolve/'.format(base_url=self._base_url)
        headers = {'Content-Type': 'application/json'}
        if self._token:
            headers['Authorization'] = 'OAuth {}'.format(self._token)
        response = json_request(
            'get', url, scheme=self.RESOLVE_INSTANCE_FILTER_SCHEME, ok_statuses=[requests.codes.ok],
            timeout=self._req_timeout, exception=ItsApiRequestException, session=self._session,
            data=json.dumps({"filter": instance_filter}), headers=headers,
        )
        return [HeadInstance(*i.split(':')) for i in response['result']]

    def get_controls(self, itags, etag=None, expect_ok=False):
        """
        Get controls values for given itags list

        :type itags: list of str
        :type etag: str
        :type expect_ok: bool
        :param expect_ok: ask ITS to return values even if
        given etag matches last version of controls
        :rtype: ItsControls
        """
        url = '{base_url}/process/'.format(base_url=self._base_url)
        headers = {
            'Content-Type': 'application/json',
            'If-None-Match': etag,
        }

        if expect_ok:
            headers['Expect'] = '200-ok'

        try:
            response = request.request(
                'post', url, ok_statuses=[requests.codes.ok, requests.codes.not_modified],
                timeout=self._req_timeout, session=self._session, data=json.dumps(itags), headers=headers
            )
        except requests.RequestException as err:
            raise ItsApiRequestException('ITS request error: {}'.format(err))

        controls = None
        if response.status_code != requests.codes.not_modified:
            try:
                controls = request.get_json_response(response, scheme=self.ITS_CONTROLS_SCHEME)
            except requests.RequestException as err:
                raise ItsApiRequestException('ITS request error: {}'.format(err))
            else:
                controls = {os.path.normpath(k): v for k, v in six.iteritems(controls)}

        cache_time = None
        cache_control = response.headers.get('Cache-Control')
        if cache_control:
            max_age = parse_dict_header(cache_control).get('max-age')
            if max_age:
                try:
                    cache_time = float(max_age)
                except ValueError:
                    pass

        etag = response.headers.get('ETag')

        return ItsControls(controls=controls, cache_time=cache_time, etag=etag, version=str(int(time.time())))

    def _get_ruchka_url(self, control_path):
        """
        Builds ITS control url.
        """
        return '{base_url}/values/{control_path}/'.format(base_url=self._base_url,
                                                          control_path=control_path.strip('/'))

    def _request(self, method, url, headers=None, ok_statuses=None, **kwargs):
        _headers = {
            'Content-Type': 'application/json',
            'Authorization': 'OAuth {}'.format(self._token) if self._token else None,
        }

        if headers is not None:
            _headers.update(headers)

        try:
            return request.request(method, url, ok_statuses=ok_statuses, session=self._session,
                                   timeout=self._req_timeout, headers=_headers, **kwargs)
        except requests.RequestException as e:
            if e.response is not None and e.response.status_code == httplib.PRECONDITION_FAILED:
                raise ItsConcurrentModificationException('ITS control concurrent modification: {}'.format(e))
            raise ItsApiRequestException('ITS request error: {}'.format(e))

    def get_control_value(self, control_path):
        """
        Gets ITS control value

        :type control_path: basestring
        :rtype: ItsControls|None
        """
        url = self._get_ruchka_url(control_path)
        response = self._request('get', url, ok_statuses=[httplib.OK, httplib.NOT_FOUND])
        if response.status_code == httplib.NOT_FOUND:
            return None
        try:
            control_dict = response.json()
        except Exception as e:
            raise ItsApiRequestException('Cannot parse ITS response: {}'.format(e))
        return ItsControl(control=control_dict, etag=response.headers.get('ETag'))

    def set_control_value(self, control_path, value, etag=None):
        """
        Sets ITS control value

        :type control_path: basestring
        :type value: basestring
        :type etag: basestring|None
        """
        url = self._get_ruchka_url(control_path)
        data = json.dumps({"value": value})
        self._request('post', url, ok_statuses=[httplib.OK], data=data, headers={'If-Match': etag})

    def delete_control_value(self, control_path, etag=None):
        """
        Deletes ITS control value

        :type control_path: basestring
        :type etag: basestring|None
        """
        url = self._get_ruchka_url(control_path)
        self._request('delete', url, ok_statuses=[httplib.OK, httplib.NO_CONTENT], headers={'If-Match': etag})
