"""
Handmade utilities to work with OTRS:
    http://wiki.yandex-team.ru/bot/api/otrs
Implementation is inspired by:
svn+ssh://arcadia.yandex.ru/arc/trunk/arcadia/yweb/webscripts/admscripts/scripts/otrs
"""
import urllib
import requests


OTRS_API_URL = 'http://bot.yandex-team.ru/api/otrs_api.php'
_TOKEN = '7efad8b8d493130fae6d2e7994488aab'


class OtrsParams(object):
    @property
    def url_params(self):
        return {}

    @property
    def form_params(self):
        return {}


class SearchParams(OtrsParams):
    def __init__(self, queue_id,
                 status=None,
                 tags=None,
                 from_=None,
                 current_page=None,
                 entries_per_page=None):
        self.queue_id = queue_id
        self.status = status
        self.tags = tags
        self.from_ = from_
        self.current_page = current_page
        self.entries_per_page = entries_per_page

    @property
    def url_params(self):
        rv = {
            'queueid': self.queue_id,
            'action': 'search'
        }
        if self.status is not None:
            rv['status'] = self.status
        if self.tags is not None:
            rv['tags'] = self.status
        if self.from_ is not None:
            rv['from'] = self.from_
        if self.current_page is not None:
            rv['current_page'] = self.current_page
        if self.entries_per_page is not None:
            rv['entries_per_page'] = self.entries_per_page
        return rv


class GetMessagesParams(OtrsParams):
    def __init__(self, queue_id,
                 ticket_number=None,
                 ticket_id=None):
        self.queue_id = queue_id
        self.ticket_number = ticket_number
        self.ticket_id = ticket_id

    @property
    def url_params(self):
        rv = {
            'queueid': self.queue_id,
            'action': 'getmessages'
        }
        if self.ticket_number is not None:
            rv['tn'] = self.ticket_number
        if self.ticket_id is not None:
            rv['tid'] = self.ticket_id
        return rv


class CloseTicketParams(OtrsParams):
    def __init__(self, ticket_id, ticket_number=None, body=None):
        self.ticked_id = ticket_id
        self.ticket_number = ticket_number
        self.body = body

    @property
    def url_params(self):
        rv = {
            'tid': self.ticked_id,
            'action': 'close',
        }
        if self.ticket_number is not None:
            rv['tn'] = self.ticket_number
        return rv

    @property
    def form_params(self):
        return {'body': self.body}


def _otrs_http_request(params, url, token, session=None):
    """
    Issue http request according to :param params:
    :type params: OtrsParams
    :type url: basestring
    :type token: basestring
    """
    # prepare url
    url_params = params.url_params
    url_params['ansformat'] = 'json'
    url_params['ansdetail'] = 'full'
    url = '{}?{}'.format(url, urllib.urlencode(url_params))
    # prepare dict for form encoding
    data = params.form_params
    data['token'] = token
    # issue request
    if session is None:
        session = requests.Session()
    resp = session.post(url, data=data)
    resp.raise_for_status()
    if resp.text == 'ERROR':
        raise Exception('OTRS returned: ERROR')
    return resp.json()


def request_tickets(params, url=OTRS_API_URL, token=_TOKEN, session=None):
    """
    Issue search tickets (ticked ids) request.
    :type params: SearchParams
    :type url: basestring
    :type token: basestring
    :rtype: dict
    """
    return _otrs_http_request(params, url, token, session=session)


def get_messages(params, url=OTRS_API_URL, token=_TOKEN, session=None):
    """
    Issue getmessages action.
    :type params: GetMessagesParams
    :type url: basestring
    :type token: basestring
    :rtype: dict
    """
    return _otrs_http_request(params, url, token, session=session)


def close_ticket(params, url=OTRS_API_URL, token=_TOKEN, session=None):
    """
    Close ticket.
    :type params: CloseTicketParams
    :type url: basestring
    :type token: basestring
    """
    return _otrs_http_request(params, url, token, session=session)


def iter_tickets(params, url=OTRS_API_URL, token=_TOKEN, session=None):
    """
    Iterate over tickets (their contents).
    :type params: SearchParams
    :type url: basestring
    :type token: basestring
    """
    # otrs has interesting API
    # we ask for pages with a list of ticket ids
    # then we ask for ticket contents themselves
    cur_page = 1
    while cur_page:
        params.current_page = cur_page
        resp = request_tickets(params, url, token, session=session)
        # navigation key can be absent if there are no open tickets
        navigation = resp.get('navigation')
        if not navigation:
            break
        cur_page = navigation['next_page']
        for ticket_id in resp['result']:
            yield ticket_id, get_messages(GetMessagesParams(params.queue_id, ticket_id=ticket_id),
                                          token=token,
                                          session=session)
