"""
Passport client which tries to reuse connections.
"""
from collections import namedtuple
from logging import getLogger

from sepelib.http import session

from .blackbox import validate_session_id, validate_oauth_token, FIELD_LOGIN, BLACKBOX_URL, BLACKBOX_AUTH_URL

log = getLogger(__name__)

PassportCheckResult = namedtuple('PassportCheckResult', ['login', 'redirect_url'])


class OAuthCheckResult(object):
    __slots__ = ['login', 'client_id', 'scope', 'error']

    def __init__(self, login, client_id, scope=None, error=None):
        self.login = login
        self.client_id = client_id
        self.scope = scope
        self.error = error

    def __str__(self):
        return "OAuthCheckResult(login='{}',client_id='{}',scope='{}',error='{}')".format(
            self.login, self.client_id, self.scope, self.error,
        )


class IPassportClient(object):
    """
    Interface to be used in dependency injection.
    """
    pass


class PassportClient(IPassportClient):
    _DEFAULT_REQ_TIMEOUT = 10

    @classmethod
    def from_config(cls, d):
        return cls(blackbox_url=d.get('blackbox_url'),
                   blackbox_auth_url=d.get('blackbox_auth_url'),
                   req_timeout=d.get('req_timeout'))

    @staticmethod
    def _get_session():
        # We do not have retries at the moment, so in order to avoid
        # having keep alive connections being dropped on IPVS while idle,
        # let us create new session every time.
        return session.InstrumentedSession('/clients/passport')

    def __init__(self, blackbox_url=BLACKBOX_URL, blackbox_auth_url=BLACKBOX_AUTH_URL, req_timeout=None):
        self._blackbox_url = blackbox_url
        self._blackbox_auth_url = blackbox_auth_url
        self._req_timeout = req_timeout if req_timeout is not None else self._DEFAULT_REQ_TIMEOUT
        if self._req_timeout < 1.0:
            raise ValueError('req_timeout must be >= 1.0, got {}'.format(self._req_timeout))

    def check_passport_cookie(self, cookies, host, user_ip, request_url):
        """
        Check passport cookie.
        :param cookies: dict-like object containing cookies,
                        e.g. :attr:`flask.request.cookies`)
        :param host: requested host, e.g. :attr:`flask.request.host`
        :param user_ip: user ip, e.g. the first element of :attr:`flask.request.access_route`
        :param request_url: URL to redirect a user to after successful authentication
        :rtype: PassportCheckResult
        """
        session_id = cookies.get('Session_id')
        if not session_id:
            return PassportCheckResult(login=None, redirect_url=self._blackbox_auth_url.format(request_url))
        if ':' in host:
            host = host.split(':')[0]

        with self._get_session() as s:
            valid, need_redirect, dbfields = validate_session_id(session_id,
                                                                 user_ip, host,
                                                                 [FIELD_LOGIN],
                                                                 timeout=self._req_timeout,
                                                                 url=self._blackbox_url,
                                                                 session=s)
        if not valid or need_redirect:
            return PassportCheckResult(login=None, redirect_url=self._blackbox_auth_url.format(request_url))
        # put login to lowercase
        # otherwise users like 'nARN' can get in trouble
        login = dbfields[FIELD_LOGIN].lower()
        log.info("authenticated via blackbox: login='{0}' ip='{1}'".format(login, user_ip))
        return PassportCheckResult(login=login, redirect_url=None)

    def check_oauth_token(self, user_ip, oauth_token=None, authorization_header=None):
        """
        Check OAuth token.
        :param oauth_token: OAuth token or entire content of Authorization header
        :param user_ip: user ip, e.g. the first element of :attr:`flask.request.access_route`
        :rtype: OAuthCheckResult
        """
        assert (oauth_token or authorization_header) and not (oauth_token and authorization_header)
        with self._get_session() as s:
            valid, fields, client_id, scope, error = validate_oauth_token(
                oauth_token=oauth_token,
                authorization_header=authorization_header,
                userip=user_ip,
                fields=[FIELD_LOGIN],
                timeout=self._req_timeout,
                url=self._blackbox_url,
                session=s)
        if valid:
            return OAuthCheckResult(login=fields[FIELD_LOGIN],
                                    client_id=client_id,
                                    scope=scope,
                                    error=None)
        else:
            return OAuthCheckResult(login=None, client_id=client_id, error=error)

    # to be consistent with other services
    def start(self):
        pass

    def stop(self):
        pass
