# coding: utf-8
import re
from six.moves.urllib.parse import urljoin

import inject

from sepelib.http.session import InstrumentedSession


class IRegistryClientFactory(object):
    @classmethod
    def instance(cls):
        """
        :rtype: IRegistryClientFactory
        """
        return inject.instance(cls)

    def get_client(self, api_url):
        """
        :type api_url: str | unicode
        :rtype: IRegistryClient
        """
        raise NotImplementedError


class RegistryClientFactory(IRegistryClientFactory):
    def __init__(self, clients=None):
        self.clients = {}
        clients = clients or []
        for config in clients:
            self.clients[config['api_url']] = RegistryClient.from_config(config)

    @classmethod
    def from_config(cls, d):
        return cls(**d)

    def get_client(self, api_url):
        return self.clients.get(api_url)


class IRegistryClient(object):
    @classmethod
    def instance(cls):
        """
        :rtype: IRegistryClient
        """
        return inject.instance(cls)

    def get_image_tags_list(self, image_name):
        """
        :type image_name: str | unicode
        :rtype: list[str]
        """
        raise NotImplementedError

    def get_image_manifest(self, image_name):
        """
        :type image_name: str | unicode
        :rtype: dict
        """
        raise NotImplementedError

    def get_layer_blob_url(self, image_name, layer_blob):
        """
        :type image_name: str | unicode
        :type layer_blob: str | unicode
        :rtype: str
        """
        raise NotImplementedError

    def get_layers_urls(self, image_name):
        """
        :type image_name: str | unicode
        :rtype: list[str]
        """
        raise NotImplementedError


class RegistryClient(IRegistryClient):
    _DEFAULT_REQ_TIMEOUT = 30
    _DEFAULT_ATTEMPTS = 3
    image_name_pattern = re.compile(r'^(?P<image>[^:@]+)(:(?P<tag>[^:@]+)|@(?P<digest>sha256:[^:@]+))$')

    def __init__(self, api_url, token, req_timeout=None, attempts=None, **kwargs):
        self.api_url = api_url
        self.token = token
        self.req_timeout = req_timeout or self._DEFAULT_REQ_TIMEOUT
        self.attempts = attempts or self._DEFAULT_ATTEMPTS
        self._session = InstrumentedSession('/clients/docker-registry')
        self._session.headers.update({
            'Authorization': 'OAuth {}'.format(token)
        })

    @classmethod
    def from_config(cls, d):
        return cls(**d)

    def _get(self, url, **kwargs):
        """
        :type url: str | unicode
        :type kwargs: dict
        :rtype: requests.Response
        """
        allow_redirects = kwargs.get('allow_redirects', True)
        response = self._session.request('get', urljoin(self.api_url, url),
                                         timeout=self.req_timeout,
                                         allow_redirects=allow_redirects)
        response.raise_for_status()
        return response

    def get_layers_urls(self, image_name):
        manifest = self.get_image_manifest(image_name)
        layers_blobs = self._get_layers_blobs_from_manifest(manifest)
        layers_urls = []
        for blob in layers_blobs:
            layer_url = self.get_layer_blob_url(image_name, blob)
            layers_urls.append(layer_url)
        return layers_urls

    def _get_layers_blobs_from_manifest(self, manifest):
        """
        :type manifest: dict
        :rtype: list[str]
        """
        for index, layer in enumerate(manifest['fsLayers']):
            if 'throwaway' not in manifest['history'][index]['v1Compatibility']:
                yield layer['blobSum']

    def get_image_tags_list(self, image_name):
        url = '/v2/{}/tags/list/'.format(image_name)
        return self._get(url).json()['tags']

    def get_image_manifest(self, image_name):
        image, tag, digest = self._parse_image_name(image_name)
        url = '/v2/{}/manifests/{}/'.format(image, tag or digest)
        return self._get(url).json()

    def get_layer_blob_url(self, image_name, layer_blob):
        image, tag, digest = self._parse_image_name(image_name)
        url = '/v2/{}/blobs/{}'.format(image, layer_blob)
        response = self._get(url, allow_redirects=False)
        return response.headers['location']

    def _parse_image_name(self, full_image_name):
        """
        :type full_image_name: str | unicode
        :rtype: tuple(str, str | None, str | None)
        """
        match = self.image_name_pattern.match(full_image_name)
        if not match:
            raise ValueError('Invalid image name {}'.format(full_image_name))
        parsed = self.image_name_pattern.match(full_image_name).groupdict()
        return parsed['image'], parsed['tag'], parsed['digest']
