"""
Sandbox (http://wiki.yandex-team.ru/Sandbox) XMLRPC API.
Original API documentation is available here: http://sandbox.yandex-team.ru/media/html/api/xmlrpc_api.html
"""
from uuid import uuid4
from six.moves.xmlrpc_client import Fault

from enum import Enum
from sepelib.http.session import InstrumentedSession
from sepelib.rpc.xml import xmlrpc_call
from sepelib.util.retry import RetrySleeper


class ResourceState(Enum):
    NOT_READY = "NOT_READY"
    READY = "READY"
    BROKEN = "BROKEN"
    DELETED = "DELETED"


class SandboxApiError(Exception):
    pass


class ISandboxClient(object):
    """
    Interface to be used in dependency injection.
    """
    pass


class SandboxClient(ISandboxClient):
    SANDBOX_API_URL = "http://sandbox.yandex-team.ru/sandbox/xmlrpc"
    DEFAULT_REQ_TIMEOUT = 10  # sec
    DEFAULT_MAX_RETRIES = 2
    DEFAULT_RETRY_TIMEOUT = 2

    @classmethod
    def from_config(cls, d):
        """
        Instantiate :class:`SandboxClient` instance from configuration dictionary.

        :param d: Configuration dictionary with parameters
        :rtype: SandboxClient
        """
        return cls(
            sandbox_api_url=d.get('sandbox_api_url'),
            oauth_token=d.get('oauth_token'),
            req_timeout=d.get('req_timeout'),
            max_retries=d.get('max_retries'))

    def __init__(self, sandbox_api_url=None, oauth_token=None, req_timeout=None, max_retries=None):
        """
        Create a :class:`SandboxClient` instance.

        :param sandbox_api_url: Override default Sandbox API URL
        :param oauth_token: Optional oauth token to be used
        :param req_timeout: Single request timeout (in seconds). Must be positive.
        :param max_retries: How many times to retry the command. Must be >= 0.
        """
        self._sandbox_api_url = sandbox_api_url or self.SANDBOX_API_URL
        self._oauth_token = oauth_token
        if req_timeout is not None and req_timeout <= 0:
            raise ValueError("Bad req_timeout value {}. Must be > 0".format(req_timeout))
        self._req_timeout = req_timeout or self.DEFAULT_REQ_TIMEOUT
        if max_retries is not None and max_retries < 0:
            raise ValueError("Bad max_retries value {}. Must be >= 0".format(max_retries))
        self._max_retries = max_retries or self.DEFAULT_MAX_RETRIES
        self._session = InstrumentedSession('/clients/sandbox')

    def _xmlrpc_call(self, method, params):
        # Sandbox server uses this to account duplicate requests.
        # Let's be nice and support this too.
        headers = {'X-Request-Id': uuid4().hex}
        if self._oauth_token:
            headers['Authorization'] = 'OAuth {}'.format(self._oauth_token)
        sleeper = RetrySleeper(max_tries=self._max_retries, max_delay=5)
        while 1:
            try:
                return xmlrpc_call(self._sandbox_api_url, method, params,
                                   timeout=self._req_timeout,
                                   exc_cls=SandboxApiError,
                                   session=self._session,
                                   headers=headers)
            except Exception as e:
                # According to official sandbox client, error code != -1 should not be retried
                if isinstance(e, Fault) and e.faultCode != -1:
                    raise
                if not sleeper.increment(exception=False):
                    raise

    def filter_resources(self, filter_):
        """
        Call Sandbox vanilla 'listResources' method.
        :param filter_: filter dictionary
        :return: list of filtered tasks
        """
        return self._xmlrpc_call('listResources', filter_)

    def list_releases(self, filter_=None):
        """
        Call vanilla Sandbox ``listReleases`` method.
        Filter is dictionary with optional attributes:
            * task_id
            * limit
            * offset
            * resource_type
            * release_status: one of: stable, testing, unstable.
        :rtype: list
        """
        if filter_ is None:
            filter_ = {}
        return self._xmlrpc_call('listReleases', filter_)

    def list_resources(self, task_id, resource_type, state='READY', filter_=None):
        """
        Call Sandbox listResources method with obligatory :param task_id: and
        optional :param resource_type: having :param state: or provide your own
        filter using :param fltr:. :return: list of task resources
        :rtype: list
        """
        if filter_ is None:
            filter_ = {}
        filter_.update(task_id=task_id,
                       resource_type=resource_type)
        if state is not None:
            filter_.update(state=state)
        return self.filter_resources(filter_)

    def list_tasks(self, filter_=None):
        """
        Call vanilla Sandbox ``listTasks`` method.
        :param filter_: filter dictionary
        :rtype: list
        """
        return self._xmlrpc_call('list_tasks', filter_)

    def get_task(self, task_id):
        """
        Call Sandbox getTask method with :param task_id:
        """
        return self._xmlrpc_call('getTask', task_id)

    def get_resource(self, resource_id):
        """
        Call Sandbox getTask method with :param task_id:
        """
        return self._xmlrpc_call('getResource', resource_id)

    def get_resource_rsync_links(self, resource_id):
        """
        Call Sandbox "getResourceRsyncLinks" method. Returns a list of rsync urls.
        :type resource_id: str or int
        :return: list
        """
        return self._xmlrpc_call('getResourceRsyncLinks', resource_id)

    def get_resource_http_links(self, resource_id):
        """
        Call Sandbox "getResourceHttpLinks" method. Returns a list of http urls.
        :type resource_id: str or int
        :return: list
        """
        return self._xmlrpc_call('getResourceHttpLinks', resource_id)

    def create_task(self, type_, owner, ctx=None, desc=None, arch=None, priority=None):
        """
        Create task in Sandbox by calling "createTask".
        :param type_: Task type
        :param owner: Owner of the task
        :param ctx: Dictionary with task specific params
        :param desc: Task description
        :param arch: Desired architecture (e.g. linux)
        :return: Task identifier
        """
        params = {
            'type_name': type_,
            'owner': owner,
        }
        if ctx:
            params['ctx'] = ctx
        if desc:
            params['descr'] = desc
        if arch:
            params['arch'] = arch
        if priority:
            params['priority'] = priority
        return self._xmlrpc_call('createTask', params)

    def create_release(self, task_id, release_status, release_subject, release_comments):
        """
        Release task in Sandbox by calling "createRelease".
        :param task_id: Task to release
        :param release_status: stable, testing or unstable
        :param release_subject: Single-line human readable message
        :param release_comments: Detailed release description
        """
        return self._xmlrpc_call('createRelease', (task_id, release_status, release_subject, release_comments))

    def cancel_task(self, task_id):
        """
        Cancel task by calling "cancelTask".
        :param task_id: Task to cancel
        :return: True or raises SandboxApiError
        """
        return self._xmlrpc_call('cancelTask', task_id)

    def start(self):
        """
        Dummy method to be compatible with other clients
        :return: None
        """
        pass

    def stop(self):
        """
        Stop all requests and close open connections.
        :return:
        """
        self._session.close()
