# -*- coding: utf-8 -*-
"""
ISS shard tracker client.
For details see https://wiki.yandex-team.ru/iss3/Specifications/ShardTracker
"""
from shard_tracker import ShardTrackerService
from shard_tracker.ttypes import NotFoundException, ShardInfo, FullShard

from sepelib.yandex import ApiRequestException
from sepelib.metrics.registry import metrics_inventory
from sepelib.rpc.thrift.client import ThriftBinClient


class ShardTrackerApiException(ApiRequestException):
    """Common-case exception for shard tracker request"""
    pass


class IShardTrackerClient(object):
    """
    Interface for dependency injection
    """
    pass


class ShardTrackerClient(IShardTrackerClient):
    _DEFAULT_HOST = 'shardtracker.search.yandex.net'
    _DEFAULT_PORT = 9100
    _DEFAULT_REQUEST_TIMEOUT = 30
    _DEFAULT_ATTEMPTS = 3

    @classmethod
    def from_config(cls, d):
        return cls(**d)

    def __init__(self, host=None, port=None, request_timeout=None, attempts=None,
                 concurrent_request_count=None, conn_maxsize=None):
        self._host = host or self._DEFAULT_HOST
        self._port = port or self._DEFAULT_PORT
        self._request_timeout = request_timeout or self._DEFAULT_REQUEST_TIMEOUT
        self._metrics = metrics_inventory.get_metrics('/clients/shard_tracker')
        self._attempts = attempts or self._DEFAULT_ATTEMPTS
        self._client = ThriftBinClient(
            client_mod=ShardTrackerService,
            host=self._host,
            port=self._port,
            request_timeout=self._request_timeout,
            max_tries=self._attempts,
            metrics=self._metrics,
            concurrent_request_count=concurrent_request_count,
            conn_maxsize=conn_maxsize,
        )

    def shard_exists(self, shard_id):
        """
        Check if shard exists

        :type shard_id: str
        :rtype: bool
        """
        try:
            self._client.getShardInfo(shard_id)
        except NotFoundException:
            return False
        except Exception as e:
            raise ShardTrackerApiException('Cannot check if shard {} exists: {}'.format(shard_id, e))
        return True

    def init_full_shard(self, shard_id, urls, size):
        """
        Initialize full (non-incremental) shard in shard tracker

        :param size: shard size in bytes (required by shard tracker)
        :type shard_id: str
        :type urls: list of str
        :type size: int
        """
        shard_info = ShardInfo(
            shardId=shard_id,
            full=FullShard(urls=urls, size=size)
        )
        try:
            self._client.initializeShard(shard_info)
        except Exception as e:
            raise ShardTrackerApiException('Cannot init shard {}: {}'.format(shard_id, e))
