# coding: utf-8
"""
http://staff-api.yandex-team.ru/v3 API wrapper.
Docs: https://beta.wiki.yandex-team.ru/staff/apiv3/
"""
import json
import six
import socket

from requests.exceptions import RequestException
from sepelib.http.session import InstrumentedSession


class StaffError(Exception):
    @property
    def url(self):
        return self.args[1]


class IStaffClient(object):
    """
    Interface to be used in dependency injection.
    """
    pass


class StaffClient(object):
    STAFF_API_URL = 'https://staff-api.yandex-team.ru/v3'
    DEFAULT_REQ_TIMEOUT = 10  # seconds
    DEFAULT_VERIFY_SSL = True

    @classmethod
    def from_config(cls, d):
        return cls(staff_url=d.get('staff_url'),
                   oauth_token=d.get('oauth_token'),
                   req_timeout=d.get('req_timeout'),
                   verify_ssl=d.get('verify_ssl'))

    def __init__(self, staff_url=None, oauth_token=None, req_timeout=None, verify_ssl=None):
        self._staff_url = staff_url or self.STAFF_API_URL
        self._token = oauth_token
        self._req_timeout = req_timeout or self.DEFAULT_REQ_TIMEOUT
        self._verify_ssl = self.DEFAULT_VERIFY_SSL if verify_ssl is None else verify_ssl
        self._session = InstrumentedSession('/clients/staff')

    def _http_get(self, url, params=None):
        try:
            headers = {
                'Authorization': 'OAuth {}'.format(self._token)
            }
            response = self._session.get(url,
                                         params=params, headers=headers,
                                         timeout=self._req_timeout, verify=self._verify_ssl)
        except socket.error as e:
            raise StaffError("socket error: {0}".format(six.text_type(e)), self._staff_url)
        except RequestException as e:
            raise StaffError('connection error: {0}'.format(six.text_type(e)), self._staff_url)
        response.raise_for_status()
        try:
            return json.loads(response.content)
        except ValueError as e:
            raise StaffError("failed to parse response: "
                             "txt='{0}' err='{1}'".format(response.content, six.text_type(e)), self._staff_url)

    def _list(self, rel_url, spec, fields, one):
        params = dict((key, value if not isinstance(value, list) else ','.join(map(str, value)))
                      for key, value in six.iteritems(spec))
        params['_fields'] = ','.join(fields)
        if one:
            params['_one'] = '1'
        return self._http_get('{}/{}/'.format(self._staff_url.rstrip('/'), rel_url.strip('/')), params)

    def list_persons(self, spec, fields=('id', 'name', 'url'), one=False):
        """
        https://staff-api.yandex-team.ru/v3/persons?_doc=1
        """
        return self._list('persons', spec, fields, one)

    def list_groups(self, spec, fields=('id', 'name'), one=False):
        """
        https://staff-api.yandex-team.ru/v3/groups?_doc=1
        """
        return self._list('groups', spec, fields, one)

    def list_groupmembership(self, spec, fields=('person',), one=False):
        """
        https://staff-api.yandex-team.ru/v3/membership?_doc=1
        """
        return self._list('groupmembership', spec, fields, one)

    def start(self):
        pass

    def stop(self):
        self._session.close()


# various helpers
def get_group_ids(staff_client, login):
    """
    Returns a list of group identifiers where provided login is a member.

    :type staff_client: StaffClient
    :type login: unicode
    :rtype: list[unicode]
    """
    spec = {
        'person.login': login,
        'group.type': 'department,service,servicerole',
        'group.is_deleted': 'false',
    }
    staff_resp = staff_client.list_groupmembership(spec=spec, fields=('group',))
    groups = staff_resp['result']
    rv = []
    for item in groups:
        # Every item in result list is a dictionary with "group" key
        group = item['group']
        t = group['type']
        if t == 'department':
            for g in group['ancestors']:
                group_id = g['id']
                rv.append(six.text_type(group_id))
            rv.append(six.text_type(group['id']))
        elif t == 'service':
            rv.append(six.text_type(group['id']))
        elif t == 'servicerole':
            rv.append(six.text_type(group['id']))
    return rv


def get_group_member_logins(staff_client, group_id):
    staff_resp = staff_client.list_groupmembership({
        'group.id': group_id,
        'person.official.is_dismissed': False
    }, fields=('person.login',))
    logins = []
    for item in staff_resp['result']:
        logins.append(item['person']['login'])
    return logins
