"""
Utilities to work with Yandex Tracker REST API:
http://wiki.yandex-team.ru/tracker/api
"""

import cgi
import json

from requests import Session, RequestException
from requests import codes
from sepelib.util.misc import drop_none


class Relationship(object):
    RELATES = "relates"
    DEPENDEND_BY = "is dependent by"
    DEPENDS_ON = "depends on"
    SUBTASK_FOR = "is subtask for"
    PARENT_TASK_FOR = "is parent task for"
    DUPLICATES = "duplicates"
    DUPLICATED_BY = "is duplicated by"
    EPIC_OF = "is epic of"
    HAS_EPIC = "has epic"


class StartrekError(Exception):
    def __init__(self, message, *args):
        message = u"Error in communication with Startrek: " + message
        super(StartrekError, self).__init__(message.format(*args) if args else message)


class StartrekConnectionError(StartrekError):
    pass


class StartrekRequestError(StartrekError):
    response = None

    def __init__(self, response, message, *args):
        self.response = response
        super(StartrekRequestError, self).__init__(message, *args)


class StartrekClient(object):
    TESTING_BASE_URL = PRODUCTION_BASE_URL = 'https://st-api.yandex-team.ru/v2'

    DEFAULT_REQ_TIMEOUT = 10

    @classmethod
    def from_config(cls, cfg):
        return cls(cfg['base_url'],
                   cfg['token'],
                   cfg.get('request_timeout'),
                   cfg.get('accept_language'))

    def __init__(self, base_url, token, request_timeout=None, accept_language=None):
        self._base_url = base_url
        self._token = token
        self._request_timeout = request_timeout or self.DEFAULT_REQ_TIMEOUT
        self._accept_language = accept_language
        self._session = Session()

    def _http_request(self, method, url, *args, **kwargs):
        url = "{}/{}".format(self._base_url, url)

        headers = kwargs.pop('headers', None) or {}
        headers["Authorization"] = "OAuth {}".format(self._token)
        if self._accept_language:
            headers["Accept-Language"] = self._accept_language
        if method.lower() == "post":
            headers["Content-Type"] = "application/javascript"

        try:
            resp = self._session.request(method, url, *args, headers=headers, **kwargs)
        except RequestException as e:
            raise StartrekConnectionError("{}", e)

        if resp.status_code == codes.no_content:
            return None

        if resp.status_code in (codes.ok, codes.created):
            try:
                return resp.json()
            except ValueError:
                raise StartrekRequestError(resp, "Got an invalid response from server.")

        content_type, _ = cgi.parse_header(resp.headers.get("Content-Type", ""))
        if content_type != "application/json":
            raise StartrekRequestError(resp, resp.reason)

        try:
            reply = resp.json()

            if (
                not isinstance(reply, dict) or
                not isinstance(reply.get("errors"), dict) or
                not isinstance(reply.get("errorMessages"), list)
            ):
                raise ValueError
        except ValueError:
            raise StartrekRequestError(resp, "Got an invalid response from server.")

        if reply["errorMessages"]:
            error = reply["errorMessages"][0]
        elif reply["errors"]:
            error = next(iter(reply["errors"].values()))
        else:
            error = "Unknown error"

        raise StartrekRequestError(resp, "{}.", error)

    def _rest_request(self, method, url, data):
        payload = json.dumps(data)
        headers = {'Content-Type': 'application/json'}
        return self._http_request(method, url, headers=headers, data=payload)

    def _rest_post(self, url, data):
        return self._rest_request("post", url, data)

    def _rest_patch(self, url, data):
        return self._rest_request("patch", url, data)

    def get_issues(self, filter=None, query=None, page=None, limit=None):
        """Get the specified issues.
        :argument query: See https://wiki.yandex-team.ru/tracker/vodstvo/query for syntax description
        :rtype: list
        """
        # request requires both json body and url-encoded params
        # according to https://wiki.yandex-team.ru/tracker/api/issues/list/
        return self._http_request(
            "post",
            "issues/_search",
            data=json.dumps(drop_none({"filter": filter, "query": query})),
            params=drop_none({"page": page, "perPage": limit})
        )

    def get_issue(self, issue_id):
        """
        Get issue :param issue_id: description from startrek.
        For possible fields in response dict please consult original documentation:
        http://wiki.yandex-team.ru/tracker/api/issues/:id#get
        :rtype: dict
        """
        url = "issues/{}".format(issue_id)
        return self._http_request('get', url)

    def create_issue(self, issue_params):
        """
        Create issue in startrek according to specification from :param issue_params:.
        For available fields please consult original documentation:
        http://wiki.yandex-team.ru/tracker/api/issues#post
        :type issue_params: dict
        :rtype: dict
        """
        url = "issues/"
        return self._rest_post(url, issue_params)

    def modify_issue(self, issue_id, issue_params):
        """Modify issue :param issue_id:."""
        self._rest_patch("issues/" + issue_id, issue_params)

    def get_relationships(self, issue_id):
        """Get all relationships of an issue.
        See https://wiki.yandex-team.ru/tracker/api/issues/links/list.
        """
        url = "issues/{}/links".format(issue_id)
        return self._http_request("get", url)

    def add_relationship(self, issue_id, relationship, other_issue_id):
        """Add relationship to the issue.
        See https://wiki.yandex-team.ru/tracker/api/issues/links/create.
        """
        self._rest_post("issues/" + issue_id + "/links", {"relationship": relationship, "issue": other_issue_id})

    def close_issue(self, issue_id, transition="close", resolution="fixed"):
        """Close :param issue_id: issue."""
        return self.execute_transition(issue_id, transition, {"resolution": resolution})

    def execute_transition(self, issue_id, transition, issue_params=None):
        """Execute requested transition for a given issue.
        Most often transition is a status change with some preset values.
        https://wiki.yandex-team.ru/tracker/api/issues/transitions-execute/
        """
        url = "issues/{issue_id}/transitions/{name}/_execute".format(issue_id=issue_id, name=transition)
        return self._rest_post(url, issue_params)

    def add_comment(self, issue_id, text=None, attachment_ids=None, summonees=None, maillist_summonees=None):
        """Add comment to issue.
        See https://wiki.yandex-team.ru/tracker/api/issues/comments/create/.
        :type issue_id: str
        :type text: str
        :type attachment_ids: list[str]
        :type summonees: list[str]
        :type maillist_summonees: list[str]
        :rtype: dict
        """
        data = drop_none({
            "text": text,
            "attachmentIds": attachment_ids,
            "summonees": summonees,
            "maillistSummonees": maillist_summonees
        })

        url = "issues/{issue_id}/comments".format(issue_id=issue_id)
        return self._rest_post(url, data)
