"""
Utilities to query opentracker.
"""
from collections import namedtuple
from six.moves.urllib.parse import urlencode
import re
import six


import bcode
import requests


ScrapeResult = namedtuple('ScrapeResult', ['complete', 'downloaded', 'incomplete'])


class Outcome(object):
    """
    Result of operations.
    It has some common case factories Success, Failure.
    """
    @classmethod
    def Success(cls):
        return cls(True)

    @classmethod
    def Failure(cls, reason, traceback=''):
        return cls(False, reason=reason, traceback=traceback)

    def asdict(self):
        return self.__dict__.copy()

    def __init__(self, is_successful, reason=None, traceback=None):
        self.is_successful = is_successful
        self.reason = reason
        self.traceback = traceback


class PeersInfo(object):
    def __init__(self, rbtorrent, outcome, result=None):
        self.rbtorrent = rbtorrent
        self.outcome = outcome
        self.result = result

    def __str__(self):
        return "PeersInfo(rbtorrent={0}, info={1})".format(self.rbtorrent, self.result)

    __repr__ = __str__


class PeersRequest(object):
    @staticmethod
    def no_info(rbtorrent):
        return PeersInfo(rbtorrent, Outcome.Failure('no information available'))

    def __init__(self, rbtorrents):
        self._rbtorrents = {}
        for rbtorrent in rbtorrents:
            self.add(rbtorrent)

    def add(self, rbtorrent):
        self._rbtorrents[rbtorrent] = PeersRequest.no_info(rbtorrent)

    def get(self, rbtorrent):
        return self._rbtorrents.get(rbtorrent)

    def set_success(self, rbtorrent, peers_info):
        self._rbtorrents[rbtorrent] = PeersInfo(rbtorrent, Outcome.Success(), result=peers_info)

    def set_failure(self, rbtorrent, reason, traceback=None):
        outcome = Outcome.Failure(reason, traceback=traceback)
        self._rbtorrents[rbtorrent] = PeersInfo(rbtorrent, outcome)

    def keys(self):
        return self._rbtorrents.keys()

    def values(self):
        return self._rbtorrents.values()

    def items(self):
        return self._rbtorrents.items()


class OpenTrackerClient(object):
    RB_TORRENT_MATCHER = re.compile('^rbtorrent:([a-z0-9]{40})$')
    TORRENTS_AT_A_TIME = 28

    def __init__(self, host,  port, timeout=10):
        self.host = host
        self.port = port
        self._timeout = timeout

    def get_peers_info(self, peers_request):
        """
        :type peers_request: PeersRequest
        """
        def iterate_in_batch(seq, count):
            """
            Small helper to iterate over :param seq: in batch each :param count: size.
            """
            it = iter(seq)
            while 1:
                result = []
                try:
                    for _ in six.moves.xrange(count):
                        result.append(next(it))
                except StopIteration:
                    yield result
                    break
                yield result

        peers_request = PeersRequest(peers_request)
        # we ask tracker in batches (because of request length constraints) about infohash states
        for batch in iterate_in_batch(peers_request.keys(), self.TORRENTS_AT_A_TIME):
            # form url for tracker
            info_hashes = []
            for rbtorrent in batch:
                m = self.RB_TORRENT_MATCHER.match(rbtorrent)
                if not m:
                    peers_request.set_failure(rbtorrent, reason="bad rbtorrent URI")
                    continue
                info_hashes.append(m.group(1).decode('hex'))
            url = "http://{host}:{port}/scrape?{cgi}".format(
                host=self.host, port=self.port, cgi=urlencode([('info_hash', i) for i in info_hashes])
            )
            try:
                resp = requests.get(url, timeout=self._timeout)
                resp.raise_for_status()
            except Exception as e:
                for i in batch:
                    peers_request.set_failure(i, reason=six.text_type(e))
            else:
                # use .content to receive raw string instead of unicode (if .text was used)
                scrape_info = bcode.bdecode(resp.content)
                # great, no we must iterate over response and
                # form rbtorrent URI back and set results
                for info_hash, info_dict in six.iteritems(scrape_info['files']):
                    rbtorrent = "rbtorrent:{0}".format(info_hash.encode('hex'))
                    scrape = ScrapeResult(**info_dict)
                    peers_request.set_success(rbtorrent, scrape)
        return peers_request
