package cmd

import (
	"log"

	"github.com/spf13/cobra"

	agentapi "a.yandex-team.ru/infra/walle/server/go/internal/api/agent"
	healthapi "a.yandex-team.ru/infra/walle/server/go/internal/api/juggler_checks"
	metricsapi "a.yandex-team.ru/infra/walle/server/go/internal/api/metrics"
	"a.yandex-team.ru/infra/walle/server/go/internal/cron"
	dmc "a.yandex-team.ru/infra/walle/server/go/internal/dmc"
	"a.yandex-team.ru/infra/walle/server/go/internal/lib"
	"a.yandex-team.ru/infra/walle/server/go/internal/monitoring"
	"a.yandex-team.ru/library/go/maxprocs"
)

var cliArgs lib.CliArgs

var rootCmd = &cobra.Command{
	Use:   "walled-go",
	Short: "Wall-E: Hardware as a Service",
	Run: func(cmd *cobra.Command, args []string) {
		log.Fatal("No app specified")
	},
}

var cronCmd = &cobra.Command{
	Use:   "cron",
	Short: "Cron service",
	Run: func(cmd *cobra.Command, args []string) {
		cron.StartApplication(&cliArgs)
	},
}

var dmcCmd = &cobra.Command{
	Use:   "dmc",
	Short: "Decision Makeing Center",
	Run: func(cmd *cobra.Command, args []string) {
		dmc.StartApplication(&cliArgs)
	},
}
var jugglerChecksCmd = &cobra.Command{
	Use:   "update-checks",
	Short: "Update juggler checks",
	Run: func(cmd *cobra.Command, args []string) {
		monitoring.UpdateChecks(&cliArgs)
	},
}

var apiHealthCmd = &cobra.Command{
	Use:   "api",
	Short: "API server for Juggler checks (health oh hosts)",
	Run: func(cmd *cobra.Command, args []string) {
		healthapi.StartApplication(&cliArgs)
	},
}

var apiMetricsCmd = &cobra.Command{
	Use:   "api-metrics",
	Short: "API server for metrics collected with cron",
	Run: func(cmd *cobra.Command, args []string) {
		metricsapi.StartApplication(&cliArgs)
	},
}

var apiAgentCmd = &cobra.Command{
	Use:   "api-agent",
	Short: "API server for Wall-E Agent requests",
	Run: func(cmd *cobra.Command, args []string) {
		agentapi.StartApplication(&cliArgs)
	},
}

func Execute() {
	maxprocs.AdjustAuto()

	if err := rootCmd.Execute(); err != nil {
		log.Fatal(err)
	}
}

func init() {
	rootCmd.PersistentFlags().StringVar(&cliArgs.RootDir, "root", ".", "Working directory")
	rootCmd.PersistentFlags().StringVar(&cliArgs.ConfigPath, "config", "", "Configuration file path")
	if err := rootCmd.MarkPersistentFlagRequired("config"); err != nil {
		log.Fatal(err)
	}
	jugglerChecksCmd.PersistentFlags().BoolVar(
		&cliArgs.DryRun,
		"dry-run",
		false,
		"Do not apply changes on Juggler server",
	)

	rootCmd.AddCommand(cronCmd)
	rootCmd.AddCommand(dmcCmd)
	rootCmd.AddCommand(jugglerChecksCmd)
	rootCmd.AddCommand(apiHealthCmd)
	rootCmd.AddCommand(apiMetricsCmd)
	rootCmd.AddCommand(apiAgentCmd)
}
