package api

import (
	"context"
	"fmt"
	"net/http"
	"os"
	"runtime/debug"
	"time"

	"github.com/labstack/echo/v4"
	"go.mongodb.org/mongo-driver/mongo"

	"a.yandex-team.ru/infra/walle/server/go/internal/lib"
	"a.yandex-team.ru/infra/walle/server/go/internal/lib/client"
	"a.yandex-team.ru/infra/walle/server/go/internal/lib/db"
	"a.yandex-team.ru/infra/walle/server/go/internal/lib/httputil"
	"a.yandex-team.ru/infra/walle/server/go/internal/lib/monitoring"
	"a.yandex-team.ru/infra/walle/server/go/internal/repos"
)

type config struct {
	BaseApp           lib.AppConfig            `yaml:",inline"`
	Mongo             db.MongoConfig           `yaml:"mongo"`
	HTTPServer        httputil.ServerConfig    `yaml:"http_server"`
	DevelopmentStands map[string]string        `yaml:"development_stands"`
	Racktables        *client.RacktablesConfig `yaml:"racktables"`
}

type application struct {
	config      *config
	base        *lib.Application
	server      *http.Server
	store       *store
	mongoClient *mongo.Client
}

func StartApplication(args *lib.CliArgs) {
	if err := os.Chdir(args.RootDir); err != nil {
		panic(err)
	}
	cnf := &config{}
	if err := lib.LoadConfig(args.ConfigPath, cnf); err != nil {
		panic(err)
	}
	app, err := initApplication(cnf)
	if err != nil {
		panic(fmt.Errorf("init app: %v", err))
	}
	if err := app.run(); err != nil {
		app.base.Logger("base").Errorf("fatal: %v", err)
		panic(err)
	}
}

func initApplication(cnf *config) (*application, error) {
	base, err := lib.NewApplication(cnf.BaseApp)
	if err != nil {
		return nil, err
	}
	ctx, cancel := context.WithTimeout(context.Background(), 10*time.Second)
	defer cancel()
	mongoDB, err := db.NewMongoConnect(ctx, cnf.Mongo)
	if err != nil {
		return nil, err
	}

	store, err := newStore(
		base.Logger("base"),
		repos.NewHostRepo(mongoDB, db.NewMongoLocalReadPreference(cnf.Mongo.GeoTags)),
		repos.NewHostNetworkRepo(mongoDB, db.NewMongoLocalReadPreference(cnf.Mongo.GeoTags)),
		repos.NewHostMacsRepo(mongoDB, db.NewMongoLocalReadPreference(cnf.Mongo.GeoTags)),
		repos.NewAuditLogRepo(mongoDB, db.NewMongoLocalReadPreference(cnf.Mongo.GeoTags)),
		repos.NewCacheRepo(mongoDB, db.NewMongoLocalReadPreference(cnf.Mongo.GeoTags)),
		client.NewRacktables(cnf.Racktables),
		cnf.DevelopmentStands,
	)
	if err != nil {
		return nil, err
	}

	e := echo.New()
	e.Use(func(next echo.HandlerFunc) echo.HandlerFunc {
		return func(ctx echo.Context) error {
			defer func() {
				if err := recover(); err != nil {
					base.Logger("base").Errorf("recover from panic: %v\n%s", err, string(debug.Stack()))
				}
			}()
			return next(ctx)
		}
	})
	e.Use(httputil.ZapLoggerMiddleware(base.Logger("base")))
	e.GET("/ping", monitoring.NewHealthCheckHTTPHandler())
	e.PUT(
		"/v1/hosts/:host_name/agent-report",
		func(ctx echo.Context) error {
			return handle(ctx, store)
		},
	)
	server, err := httputil.NewServer(cnf.HTTPServer, e)
	if err != nil {
		return nil, err
	}
	return &application{
		base:        base,
		config:      cnf,
		server:      server,
		store:       store,
		mongoClient: mongoDB.Client(),
	}, nil
}

func (app *application) run() error {
	defer app.close()
	app.base.AddServeFunc("http", httputil.NewServeFunc(app.config.HTTPServer, app.server))
	app.base.AddServeFunc("db-writer", app.store.writeData)
	app.base.AddServeFunc("dev-stands", app.store.updateDevStands)
	return app.base.Run()
}

func (app *application) close() {
	ctx, cancel := context.WithTimeout(context.Background(), 5*time.Second)
	defer cancel()
	_ = app.mongoClient.Disconnect(ctx)
}
