package api

import (
	"context"
	"encoding/json"
	"errors"
	"fmt"
	"net/http"
	"time"

	"github.com/labstack/echo/v4"

	"a.yandex-team.ru/infra/walle/server/go/internal/lib/automation"
	"a.yandex-team.ru/infra/walle/server/go/internal/lib/httputil"
	netutil "a.yandex-team.ru/infra/walle/server/go/internal/lib/net"
	"a.yandex-team.ru/infra/walle/server/go/internal/repos"
)

const successMessage = "The report has been processed."

var errUnknownActiveMAC = errors.New("unknown active MAC")

type Store interface {
	GetDevStands(hostName string) []string
	SaveHostMACs(report *agentReport, hostName string)
	GetHost(ctx context.Context, hostName string) (*host, error)
	GetOrCreateNetwork(ctx context.Context, hostUUID string) (*repos.HostNetwork, error)
	SaveActiveMAC(report *agentReport, host *host, network *repos.HostNetwork)
	SaveIPs(report *agentReport, host *host, network *repos.HostNetwork)
	SaveSwitches(report *agentReport, host *host, network *repos.HostNetwork)
	SaveAgentVersion(report *agentReport, host *host)
	SaveAgentErrors(report *agentReport, host *host)
}

type agentReport struct {
	MACs     map[netutil.MAC]bool    `json:"macs"` // { MAC: isActive }
	IPs      []netutil.IP            `json:"ips"`
	Switches []*networkSwitch        `json:"switches"`
	Errors   []string                `json:"errors"`
	Version  repos.WalleAgentVersion `json:"version"`

	activeMACs []netutil.MAC
	macList    []netutil.MAC
	timestamp  time.Time
	ctx        context.Context
}

func (r *agentReport) preprocess() {
	for mac, active := range r.MACs {
		r.macList = append(r.macList, mac)
		if active {
			r.activeMACs = append(r.activeMACs, mac)
		}
	}
}

func (r *agentReport) process(s Store, hostName string) error {
	r.preprocess()
	s.SaveHostMACs(r, hostName)

	host, err := s.GetHost(r.ctx, hostName)
	if err != nil {
		return err
	}
	network, err := s.GetOrCreateNetwork(r.ctx, host.uuid)
	if err != nil {
		return err
	}
	if !host.checkActiveMAC(r.activeMACs) {
		return errUnknownActiveMAC
	}
	s.SaveActiveMAC(r, host, network)
	s.SaveIPs(r, host, network)
	s.SaveSwitches(r, host, network)
	s.SaveAgentVersion(r, host)
	s.SaveAgentErrors(r, host)
	return nil
}

type networkSwitch struct {
	Name netutil.SwitchName `json:"switch"`
	Port netutil.SwitchPort `json:"port"`
	Time int64              `json:"time"` // Actualization time
}

type response struct {
	Result      string   `json:"result"`
	OtherStands []string `json:"other_stands,omitempty"`
}

func handle(ctx echo.Context, store Store) error {
	// TODO: authentication
	report := &agentReport{}
	if err := ctx.Bind(report); err != nil {
		return httputil.RespondError(ctx, http.StatusBadRequest, err, "parse request")
	}
	report.timestamp = time.Now()
	report.ctx = ctx.Request().Context()

	hostName := ctx.Param("host_name")
	if err := validate(hostName, report); err != nil {
		return httputil.RespondError(ctx, http.StatusBadRequest, err, "validate report")
	}
	resp := response{
		Result:      successMessage,
		OtherStands: store.GetDevStands(hostName),
	}

	if err := report.process(store, hostName); err != nil {
		resp.Result = err.Error()
		switch err {
		case errHostNotFound, errUnknownActiveMAC:
			resp.Result = fmt.Sprintf("Ignore report: %s", err.Error())
		default:
			return httputil.RespondError(ctx, http.StatusInternalServerError, err, "process report")
		}
	}
	b, err := json.Marshal(&resp)
	if err != nil {
		return httputil.RespondError(ctx, http.StatusInternalServerError, err, "marshal response")
	}
	return ctx.Blob(http.StatusOK, echo.MIMEApplicationJSON, b)
}

func validate(hostName string, report *agentReport) error {
	if !netutil.IsValidFQDN(hostName) {
		return errors.New("invalid FQDN")
	}
	if report.Version == "" {
		return errors.New("version required")
	}
	for _, s := range report.Switches {
		if s.Name == "" || s.Port == "" {
			return errors.New("switch name and port required")
		}
	}
	return nil
}

type host struct {
	uuid         string
	inv          int
	name         string
	project      string
	macs         []netutil.MAC
	ips          []netutil.IP
	location     *repos.HostLocation
	state        automation.HostStatus
	agentVersion repos.WalleAgentVersion
}

func (h *host) checkActiveMAC(activeMACs []netutil.MAC) bool {
	if len(activeMACs) == 0 {
		return true
	}
	for _, active := range activeMACs {
		for _, savedMAC := range h.macs {
			if savedMAC == active {
				return true
			}
		}
	}
	return false
}
